/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * boundary.c
 *
 * compute potential and forces due to boundary conditions
 */

#include <stdlib.h>
#include <math.h>
#include "force/force.h"
#include "debug/debug.h"

static int sphere(Force *f, const MD_Dvec *pos, const MD_Dvec *wrap);
static int cylinder(Force *f, const MD_Dvec *pos, const MD_Dvec *wrap);


int force_setup_boundary(Force *f)
{
  ForceParam *p = f->param;
  const int32 flags = p->flags;

  ASSERT(flags & FORCE_MASK_BC);

  f->is_bcterm1 = (p->konst1 != 0.0);
  f->is_bcterm2 = (p->konst2 != 0.0);

  /* make sure that term1 constants are positive */
  if (f->is_bcterm1
      && (p->radius1 <= 0.0 || p->exp1 <= 0
        || ((flags & FORCE_MASK_BC) != FORCE_SPHERE && p->length1 <= 0.0))) {
    return FORCE_FAIL;
  }

  /* make sure that term2 constants are positive */
  if (f->is_bcterm2
      && (p->radius2 <= 0.0 || p->exp2 <= 0
        || ((flags & FORCE_MASK_BC) != FORCE_SPHERE && p->length2 <= 0.0))) {
    return FORCE_FAIL;
  }

  if (f->is_bcterm1 && f->is_bcterm2) {

    /* determine square of smallest radius */
    if (p->radius1 < p->radius2) {
      f->sq_minradius = p->radius1 * p->radius1;
    }
    else {
      f->sq_minradius = p->radius2 * p->radius2;
    }

    /* determine smallest length */
    if (p->length1 < p->length2) {
      f->minlength = p->length1;
    }
    else {
      f->minlength = p->length2;
    }
  }
  else if (f->is_bcterm1) {
    f->sq_minradius = p->radius1 * p->radius1;
    f->minlength = p->length1;
  }
  else if (f->is_bcterm2) {
    f->sq_minradius = p->radius2 * p->radius2;
    f->minlength = p->length2;
  }
  else {
    /* neither term1 nor term2 would be active */
    return FORCE_FAIL;
  }

  return 0;
}


int force_compute_boundary(Force *f, const MD_Dvec *pos, const MD_Dvec *wrap)
{
  const int32 flags = f->param->flags;

  ASSERT(pos != NULL);
  ASSERT(wrap != NULL);
  ASSERT((flags & FORCE_MASK_BC) == FORCE_SPHERE
      || (flags & FORCE_MASK_BC) == FORCE_X_CYLINDER
      || (flags & FORCE_MASK_BC) == FORCE_Y_CYLINDER
      || (flags & FORCE_MASK_BC) == FORCE_Z_CYLINDER);

  if (flags & FORCE_SPHERE) {
    /* compute spherical boundary conditions */
    return sphere(f, pos, wrap);
  }
  else {
    /* compute cylindrical boundary conditions */
    return cylinder(f, pos, wrap);
  }
}


/*
 * spherical boundary conditions
 * (sphere about center)
 */
int sphere(Force *f, const MD_Dvec *pos, const MD_Dvec *wrap)
{
  const MD_Dvec center = f->param->center;
  const double radius1 = f->param->radius1;
  const double radius2 = f->param->radius2;
  const double konst1 = f->param->konst1;
  const double konst2 = f->param->konst2;
  const int32 exp1 = f->param->exp1;
  const int32 exp2 = f->param->exp2;
  const double sq_minradius = f->sq_minradius;
  double dx, dy, dz, dis, sq_dis, diff, factor, fc;
  double pe = 0.0;
  MD_Dvec *fb = (f->result->f_bound ? f->result->f_bound : f->result->f);
  const int32 natoms = f->param->atom_len;
  const int32 is_bcterm1 = f->is_bcterm1;
  const int32 is_bcterm2 = f->is_bcterm2;
  int32 n, k;

  ASSERT(radius1 >= 0.0);
  ASSERT(radius2 >= 0.0);
  ASSERT(is_bcterm1 || is_bcterm2);
  ASSERT(f->param->flags & FORCE_SPHERE);

  for (n = 0;  n < natoms;  n++) {
    dx = center.x - (pos[n].x + wrap[n].x);
    dy = center.y - (pos[n].y + wrap[n].y);
    dz = center.z - (pos[n].z + wrap[n].z);
    sq_dis = dx * dx + dy * dy + dz * dz;
    if (sq_dis <= sq_minradius) continue;
    dis = sqrt(sq_dis);
    fc = 0.0;
    if (is_bcterm1 && dis > radius1) {
      diff = dis - radius1;
      factor = konst1;
      for (k = 1;  k < exp1;  k++)  factor *= diff;
      pe += factor * diff;
      fc += exp1 * factor;
    }
    if (is_bcterm2 && dis > radius2) {
      diff = dis - radius2;
      factor = konst2;
      for (k = 1;  k < exp2;  k++)  factor *= diff;
      pe += factor * diff;
      fc += exp2 * factor;
    }
    fc /= dis;
    fb[n].x += fc * dx;
    fb[n].y += fc * dy;
    fb[n].z += fc * dz;
  }
  f->energy->bound = pe;
  return 0;
}


/*
 * cylindrical boundary conditions
 * (cylinder about center, parallel to either x, y, or z axis)
 */
int cylinder(Force *f, const MD_Dvec *pos, const MD_Dvec *wrap)
{
  const MD_Dvec center = f->param->center;
  const double radius1 = f->param->radius1;
  const double radius2 = f->param->radius2;
  const double length1 = f->param->length1;
  const double length2 = f->param->length2;
  const double konst1 = f->param->konst1;
  const double konst2 = f->param->konst2;
  const int32 exp1 = f->param->exp1;
  const int32 exp2 = f->param->exp2;
  const double sq_minradius = f->sq_minradius;
  const double minlength = f->minlength;
  double dx, dy, dz, dis, sq_dis, diff, factor, fr, fl, len;
  double pe = 0.0;
  MD_Dvec *fb = (f->result->f_bound ? f->result->f_bound : f->result->f);
  const int32 flags = (f->param->flags & FORCE_MASK_BC);
  const int32 natoms = f->param->atom_len;
  const int32 is_bcterm1 = f->is_bcterm1;
  const int32 is_bcterm2 = f->is_bcterm2;
  int32 n, k, is_neg;

  ASSERT(radius1 >= 0.0);
  ASSERT(radius2 >= 0.0);
  ASSERT(is_bcterm1 || is_bcterm2);
  ASSERT(flags == FORCE_X_CYLINDER
      || flags == FORCE_Y_CYLINDER
      || flags == FORCE_Z_CYLINDER);

  for (n = 0;  n < natoms;  n++) {
    dx = center.x - (pos[n].x + wrap[n].x);
    dy = center.y - (pos[n].y + wrap[n].y);
    dz = center.z - (pos[n].z + wrap[n].z);
    if (flags == FORCE_X_CYLINDER) {
      is_neg = (dx < 0.0);
      len = (is_neg ? -dx : dx);
      sq_dis = dy * dy + dz * dz;
    }
    else if (flags == FORCE_Y_CYLINDER) {
      is_neg = (dy < 0.0);
      len = (is_neg ? -dy : dy);
      sq_dis = dx * dx + dz * dz;
    }
    else {
      is_neg = (dz < 0.0);
      len = (is_neg ? -dz : dz);
      sq_dis = dx * dx + dy * dy;
    }
    fl = 0.0;
    if (len > minlength) { 
      if (is_bcterm1 && len > length1) {
        diff = len - length1;
        factor = konst1;
        for (k = 1;  k < exp1;  k++)  factor *= diff;
        pe += factor * diff;
        fl += exp1 * factor;
      }
      if (is_bcterm2 && len > length2) {
        diff = len - length2;
        factor = konst2;
        for (k = 1;  k < exp2;  k++)  factor *= diff;
        pe += factor * diff;
        fl += exp2 * factor;
      }
      if (is_neg) fl = -fl;
    }
    fr = 0.0;
    if (sq_dis > sq_minradius) {
      dis = sqrt(sq_dis);
      if (is_bcterm1 && dis > radius1) {
        diff = dis - radius1;
        factor = konst1;
        for (k = 1;  k < exp1;  k++)  factor *= diff;
        pe += factor * diff;
        fr += exp1 * factor;
      }
      if (is_bcterm2 && dis > radius2) {
        diff = dis - radius2;
        factor = konst2;
        for (k = 1;  k < exp2;  k++)  factor *= diff;
        pe += factor * diff;
        fr += exp2 * factor;
      }
      fr /= dis;
    }
    if (flags == FORCE_X_CYLINDER) {
      fb[n].x += fl;
      fb[n].y += fr * dy;
      fb[n].z += fr * dz;
    }
    else if (flags == FORCE_Y_CYLINDER) {
      fb[n].x += fr * dx;
      fb[n].y += fl;
      fb[n].z += fr * dz;
    }
    else {
      fb[n].x += fr * dx;
      fb[n].y += fr * dy;
      fb[n].z += fl;
    }
  }
  f->energy->bound = pe;
  return 0;
}
