/*
 * Copyright (C) 2003-2005 by David J. Hardy.  All rights reserved.
 *
 * debug.h
 *
 * Note that the use of
 *
 *   do { ... } while (0)
 *
 * below forces macro calls to be terminated with a semicolon.
 * Best to do this whenever bracket grouping { } is needed within
 * a macro definition.
 */

/**@file    debug.h
 * @brief   Collection of debugging macros.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * This file contains a collection of debugging macros.
 * It is intended to be included in C source files only,
 * not in C header files.
 *
 * The macros @c ERRMSG() and @c BUG() are available by simply
 * including this header file.
 * The macros @c ASSERT(), @c COND(), and @c ERRCOND() provide
 * assertion checking that is enabled by defining
 * either @c DEBUG_SUPPORT or @c DEBUG_WATCH.
 * Defining @c DEBUG_WATCH permits the expansion of
 * macros @c TEXT(), @c STR(), @c INT(), @c PTR(), @c FLT(), and @c VEC()
 * that displays quantities.
 * A non-externally linked global integer variable @c DEBUG_COUNTER
 * is also enabled by defining @c DEBUG_WATCH.
 * The @c DEBUG_COUNTER variable is automatically incremented by
 * each of the "watch" macros.
 * It can be useful in the context of debugging loops.
 */

#ifndef DEBUG_H
#define DEBUG_H

#include <stdio.h>
#include <stdlib.h>

/*
 * ERRMSG(msg) - print nil-terminated msg string to stderr
 */
#define ERRMSG(msg) \
  fprintf(stderr, "ERROR: %s, line %d: %s\n", __FILE__, __LINE__, msg)

/*
 * BUG(msg) - print nil-terminated msg string to stderr and abort
 *
 * Use for bug catching, for things that should never go wrong
 * (activated whether or not debugging support is on).
 */
#define BUG(msg) \
  do { ERRMSG(msg); abort(); } while (0)


#if defined(DEBUG_SUPPORT) || defined(DEBUG_WATCH)

#include <string.h>
#include <errno.h>

/*
 * ASSERT(expr) - assert macro, expr is evaluated once as (boolean) integer
 *
 * Tests to see if asserted condition is true.  If it *is not* true, print
 * error diagnostics to stderr and abort.
 */
#define ASSERT(expr) \
do { \
  if ( !(expr) ) { \
    fprintf(stderr, "ASSERTION FAILED: %s, line %d: \"%s\"\n", \
        __FILE__, __LINE__, #expr); \
    abort(); \
  } \
} while (0)

/*
 * COND(expr) - exceptional condition macro, expr is evaluated once as
 *   (boolean) integer
 *
 * Tests to catch an exception.  If condition *is* true, print error
 * diagnostics to stderr and abort.  (Opposite behavior of ASSERT().)
 */
#define COND(expr) \
do { \
  if ( expr ) { \
    fprintf(stderr, "EXCEPTIONAL CONDITION: %s, line %d: \"%s\"\n", \
        __FILE__, __LINE__, #expr); \
    abort(); \
  } \
} while (0)

/*
 * ERRCOND(expr) - exceptional condition macro, expr is evaluated once
 *   as (boolean) integer, also calls strerror(errno) for diagnostics.
 *
 * Tests to catch an exception from C system library call.  Same behavior
 * as COND() but diagnostics also returns strerror(errno) message.
 */
#define ERRCOND(expr) \
do { \
  if ( expr ) { \
    fprintf(stderr, "EXCEPTIONAL CONDITION: %s, line %d: \"%s\"\n" \
        "SYSTEM ERROR: %s\n", __FILE__, __LINE__, #expr, \
        (strerror(errno) ? strerror(errno) : "")); \
    abort(); \
  } \
} while (0)

#ifdef DEBUG_WATCH

/* keep count of debug watches (below) */
static int DEBUG_COUNTER = 0;

/* can customize debug watches by redefining title */
#ifndef DEBUG_TITLE
#define DEBUG_TITLE "DEBUG"
#endif

/*
 * TEXT(msg) - print text message to stderr
 *
 * Argument should be a nil-terminated string literal.
 */
#define TEXT(t) \
fprintf(stderr, DEBUG_TITLE " %d (%s, line %d) text:  %s\n", \
    DEBUG_COUNTER++, __FILE__, __LINE__, t)

/*
 * STR(s) - print to stderr value from string variable
 *
 * Argument should be a variable of type char * that points to a
 * nil-terminated string (or pointer could be NULL).
 */
#define STR(str) \
do { \
  const char *DEBUG_str = (str); \
  (DEBUG_str ? \
  fprintf(stderr, DEBUG_TITLE " %d (%s, line %d) string:  (%s)=\"%s\"\n", \
      DEBUG_COUNTER++, __FILE__, __LINE__, #str, DEBUG_str) : \
  fprintf(stderr, DEBUG_TITLE " %d (%s, line %d) string:  (%s)=(nil)\n", \
      DEBUG_COUNTER++, __FILE__, __LINE__, #str)); \
} while (0)

/*
 * INT(i) - print to stderr value from an integer expression
 */
#define INT(i) \
fprintf(stderr, DEBUG_TITLE " %d (%s, line %d) int expr: (%s)=%d\n", \
    DEBUG_COUNTER++, __FILE__, __LINE__, #i, (int)(i))

/*
 * PTR(p) - print to stderr address of pointer (evaluate as hex expression)
 */
#define PTR(p) \
fprintf(stderr, DEBUG_TITLE " %d (%s, line %d) hex addr: (%s)=%p\n", \
    DEBUG_COUNTER++, __FILE__, __LINE__, #p, (void *)(p))

/*
 * FLT(x) - print to stderr value from a float or double expression
 */
#define FLT(x) \
fprintf(stderr, DEBUG_TITLE " %d (%s, line %d) fp expr:  (%s)=%g\n", \
          DEBUG_COUNTER++, __FILE__, __LINE__, #x, (double)(x))

/*
 * VEC(v) - print to stderr component values of 3D vector struct
 *   (like mdtypes.h Fvec or Dvec)
 *
 * Note that since the argument is a struct, it does not matter that
 * it is evaluated multiple times.
 */
#define VEC(v) \
fprintf(stderr, DEBUG_TITLE " %d (%s, line %d) vec var:  (%s)=(%g %g %g)\n", \
    DEBUG_COUNTER++, __FILE__, __LINE__, #v, \
    (double)((v).x), (double)((v).y), (double)((v).z))

#else /* DEBUG_WATCH off */

#define TEXT(t)
#define STR(s)
#define INT(i)
#define PTR(p)
#define FLT(x)
#define VEC(v)

#endif /* DEBUG_WATCH */

#else /* DEBUG_SUPPORT off */

#define ASSERT(expr)
#define COND(expr)
#define ERRCOND(expr)

#define TEXT(t)
#define STR(s)
#define INT(i)
#define PTR(p)
#define FLT(x)
#define VEC(v)

#endif /* DEBUG_SUPPORT */

#endif /* DEBUG_H */
