/*
 * Copyright (C) 2003-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    table.h
 * @brief   Association table.
 * @author  John Stone and David J. Hardy
 * @date    2003-2005
 *
 * The @c adt_Table class provides an association table, a mapping of keys
 * to values (i.e. any key has exactly one value associated with it).
 * The keys are nil-terminated strings and the values are nonnegative
 * integers (most likely an array index).  The @c adt_Table supports fast
 * @f$ O(1) @f$ searching by applying a hash function to the key.
 *
 * The integer value associated with a key is intended to be an array index
 * (i.e. nonnegative).  Otherwise, the value will conflict with the
 * @c ADT_ERROR (-1) return value indicating an error.  
 * The @c const @c char @c * strings passed in as keys are aliased by the
 * table, so they must persist at least as long as the table does.
 * The easiest way is to use string literals as hash keys.
 *
 * Errors are generally indicated by a return value of @c ADT_ERROR,
 * due either to failed memory allocation or inability to find a key in
 * the table, depending on the function.
 * 
 * The hash table implementation was donated by John Stone,
 * and the interface was subsequently modified by David Hardy.
 */

#ifndef ADT_TABLE_H
#define ADT_TABLE_H

#ifdef __cplusplus
extern "C" {
#endif

#ifndef ADT_ERROR
#define ADT_ERROR  (-1)  /**< Return value from failed function call. */
#endif

  /**@brief Table container class.
   *
   * Members should be treated as private.
   */
  typedef struct adt_Table_t {
    struct hash_node_t **bucket;  /* array of hash nodes */
    int size;                     /* size of the array */
    int entries;                  /* number of entries in table */
    int downshift;                /* shift cound, used for hashing */
    int mask;                     /* used to select bits for hashing */
    char buf[64];                 /* used to store buffer for stats */
  } adt_Table;


  /**@brief Constructor.
   *
   * Creates dynamically allocated table object.
   * @param[in] size  The starting number of table entries.
   *
   * Note that the table will still be dynamically resized if enough
   * insertions are performed.  A good estimate for @p size can avoid
   * resizing, roughly twice the number of expected table entries.
   *
   * @return Pointer to table object or @c NULL if memory allocation
   * fails.
   */
  adt_Table *adt_createTable(int size);


  /**@brief Destructor.
   *
   * Frees dynamically allocated table space and dynamically allocated
   * table object.
   */
  void adt_destroyTable(adt_Table *);


  /**@brief Insert (@c key, @c value) pair into table.
   *
   * @param[in] key    A nil-terminated string.
   * @param[in] value  A nonnegative integer.
   *
   * Make sure that the string pointed to by @p key persists until after
   * the table is destroyed.  The associated @p value is probably an
   * array index.
   *
   * This routine does not update an existing value for @p key if @p key
   * is already in the table.  Instead call @p adt_updateTable().
   *
   * @return The @p value argument on success.
   * If @p key is already present in the table, then the value
   * already associated with it is instead returned.
   * If memory cannot be allocated for the new (@c key, @c value) pair,
   * then @c ADT_ERROR is returned.
   *
   * Note that a memory allocation error fails gracefully (i.e. you can
   * try freeing other memory from the heap and then re-invoke this
   * function).  An error leaves the state of the existing table intact.
   */
  int adt_insertTable(adt_Table *, const char *key, int value);


  /**@brief Lookup table entry value associated with given key.
   *
   * @param[in] key  A nil-terminated string.
   *
   * @return The nonnegative value associated with @p key
   * of @c ADT_ERROR if key is not in the table.
   */
  int adt_lookupTable(adt_Table *, const char *key);


  /**@brief Update value for table entry.
   *
   * @param[in] key       A nil-terminated string.
   * @param[in] newvalue  A nonnegative integer.
   *
   * This changes the (@c key, @c value) association in place:
   * @f[
   *   (\mathtt{key},\mathtt{oldvalue}) \rightarrow
   *   (\mathtt{key},\mathtt{newvalue}).
   * @f]
   *
   * \return The previous value that had been associated with @p key
   * or @c ADT_ERROR if @p key is not in the table.
   */
  int adt_updateTable(adt_Table *, const char *key, int newvalue);


  /**@brief Delete key and its associated value from table.
   *
   * @param[in] key   A nil-terminated string.
   *
   * \return The nonnegative value that had been associated with @p key
   * or @p ADT_ERROR if @p key is not in the table.
   */
  int adt_deleteTable(adt_Table *, const char *key);


  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated table object.
   * See @c adt_createTable() for a description of expected arguments.
   */
  int adt_initializeTable(adt_Table *, int size);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated table object
   * (i.e. one constructed using @c adt_initializeTable() ).
   */
  void adt_cleanupTable(adt_Table *);


  /**@brief Compute table statistics.
   *
   * @return A string description "# slots, # entries, and # ALOS"
   * (ALOS = average length of search) providing diagnostics for
   * the data structure performance.
   */
  const char *adt_getStatsTable(adt_Table *);

#ifdef __cplusplus
}
#endif

#endif /* ADT_TABLE_H */
