/*
 * Copyright (C) 2003-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    list.h
 * @brief   List container.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c adt_List class provides a container for an array-based list.
 * It essentially repeats the @c adt_Array class semantics and extends
 * it with operations that append (@c adt_appendList() ) and delete
 * (@c adt_deleteList() ) a single element to and from the end of the array.
 * These operations are implemented so as to provide amortized cost
 * of @f$ O(1) @f$ per operation.  This container is ideal for
 * incrementally building an array.
 *
 * Functions returning @c int generally indicate success by returning 0
 * and failure by returning @c ADT_ERROR.  The access functions that accept
 * an array index will terminate the program if the index is out-of-range.
 * Error-checking is necessary only for calls to the constructor
 * @c adt_createList() (or alternatively @c adt_initializeList() ),
 * to the resize function @c adt_resizeList(), and to the two list-specific
 * operations @c adt_appendList() and @c adt_deleteList().
 */

#ifndef ADT_LIST_H
#define ADT_LIST_H

#ifdef __cplusplus
extern "C" {
#endif

#include "adt/array.h"

#ifndef ADT_ERROR
#define ADT_ERROR  (-1)  /**< Return value from failed function call. */
#endif

  /**@brief List container class.
   *
   * Members should be treated as private.
   */
  typedef struct adt_List_t {
    adt_Array array;
    int len;
  } adt_List;


  /**@brief Constructor.
   *
   * Creates dynamically allocated list object.
   * @param[in] elemsz  The size in bytes of list elements, must be positive.
   * @param[in] len  The length of list measured in number of elements,
   *   must be nonnegative.
   * @param[in] buffer  Provides a preallocated buffer space to be used.
   *
   * If @p buffer is @c NULL, then the array buffer space is dynamically
   * allocated and can be resized.
   *
   * If @p buffer is not @c NULL, then the space provided must be at
   * least @p elemsz times @p len bytes.  The user retains control over
   * buffer space provided (e.g. user must free dynamic allocations).
   *
   * @return Pointer to array object or @c NULL if the parameters are
   * inconsistent or if memory allocation fails.
   */
  adt_List *adt_createList(int elemsz, int len, void *buffer);


  /**@brief Destructor.
   *
   * Frees dynamically allocated buffer space and dynamically allocated
   * list object.
   */
  void adt_destroyList(adt_List *);


  /**@brief Returns length of list. */
  int adt_getLengthList(adt_List *);


  /**@brief Returns list element size in bytes. */
  int adt_getElemsizeList(adt_List *);


  /**@brief Returns pointer to list buffer space.
   *
   * This permits direct access of list data for efficiency.
   * Note that resizing the list will likely change the
   * memory location of the buffer, invalidating earlier calls
   * to this function.
   */
  void *adt_getDataList(adt_List *);


  /**@brief Append element to list.
   *
   * @param[in] pelem  Points to the desired element value.
   *
   * Extends the list length by one and performs a deep copy of
   * the value located at @p elem into the last element location.
   *
   * @return 0 on success or @c ADT_ERROR on failure.
   */
  int adt_appendList(adt_List *, void *pelem);


  /**@brief Delete element from list.
   *
   * Removes the last element from the list and decreases the length
   * of the list by one.  Attempting to remove an element from a list
   * of length 0 is treated as a bug that  terminates the program.
   *
   * @return 0 on success or @c ADT_ERROR on failure.
   */
  int adt_deleteList(adt_List *);


  /**@brief Index the list.
   *
   * @param[in] index  The list index using zero-based indexing.
   *
   * Must have @f$ 0 \leq @f$ @p index @f$ < @f$ length.
   * An @p index out-of-range will terminate program with error message.
   *
   * @return Pointer to the indicated list element.  Typecast to the
   * actual element type in order to use it.
   */
  void *adt_indexList(adt_List *, int index);


  /**@brief Update a list element.
   *
   * @param[in] index  The list index using zero-based indexing.
   * @param[in] pelem  Points to the desired element value.
   *
   * Must have @f$ 0 \leq @f$ @p index @f$ < @f$ length.
   * An @p index out-of-range will terminate program with error message.
   *
   * Performs a deep copy of the value located at @p pelem
   * into the indicated list element position.
   *
   * @return 0 (does not fail).
   */
  int adt_updateList(adt_List *, int index, void *pelem);


  /**@brief Resize the list.
   *
   * @param[in] len  The new length for the list.
   *
   * Must have @f$ 0 \leq @f$ @p len.
   * Fails if memory allocation is not possible.
   * (Terminates program as a bug if @p len is illegal or if an attempt is
   * made to resize a user-provided buffer.)
   *
   * @return 0 on success or @c ADT_ERROR on failure.
   */
  int adt_resizeList(adt_List *, int len);


  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated list object.
   * See @c adt_createList() for a description of expected arguments.
   */
  int adt_initializeList(adt_List *, int elemsz, int len, void *buffer);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated list object
   * (i.e. one constructed using @c adt_initializeList() )
   */
  void adt_cleanupList(adt_List *);

#ifdef __cplusplus
}
#endif

#endif /* ADT_LIST_H */
