/*
 * Copyright (C) 2003-2005 by David J. Hardy.  All rights reserved.
 *
 * list.c
 */

#include <stdlib.h>
#include "adt/list.h"
#include "debug/debug.h"


enum {
  MINLEN_BUFFER = 5
};


adt_List *adt_createList(int elemsz, int len, void *buffer)
{
  adt_List *p;
  p = (adt_List *) malloc(sizeof(adt_List));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (adt_initializeList(p, elemsz, len, buffer)) {
    free(p);
    return NULL;
  }
  return p;
}


int adt_initializeList(adt_List *p, int elemsz, int len, void *buffer)
{
  int initarraylen = len;
  ASSERT(p != NULL);
  if (buffer == NULL && initarraylen == 0) {
    initarraylen = MINLEN_BUFFER;
  }
  if (adt_initializeArray(&(p->array), elemsz, initarraylen, buffer)) {
    ERRMSG("unable to initialize array");
    return ADT_ERROR;
  }
  p->len = len;
  return 0;
}


void adt_destroyList(adt_List *p)
{
  ASSERT(p != NULL);
  adt_cleanupList(p);
  free(p);
}


void adt_cleanupList(adt_List *p)
{
  ASSERT(p != NULL);
  adt_cleanupArray(&(p->array));
}


int adt_getLengthList(adt_List *p)
{
  ASSERT(p != NULL);
  return p->len;
}


int adt_getElemsizeList(adt_List *p)
{
  ASSERT(p != NULL);
  return adt_getElemsizeArray(&(p->array));
}


void *adt_getDataList(adt_List *p)
{
  ASSERT(p != NULL);
  return adt_getDataArray(&(p->array));
}


int adt_appendList(adt_List *p, void *pelem)
{
  ASSERT(p != NULL);
  if (adt_getLengthArray(&(p->array)) == p->len) {
    if (adt_resizeArray(&(p->array), 2 * p->len)) {
      ERRMSG("cannot resize array");
      return ADT_ERROR;
    }
  }
  if (pelem != NULL) {
    if (adt_updateArray(&(p->array), p->len, pelem)) {
      ERRMSG("cannot update array element");
      return ADT_ERROR;
    }
  }
  p->len++;
  return 0;
}


int adt_deleteList(adt_List *p)
{
  int arraylen;
  ASSERT(p != NULL);
  if (p->len == 0) {
    BUG("cannot delete elements from an empty list");
  }
  arraylen = adt_getLengthArray(&(p->array));
  if ((p->len <= arraylen / 4) && (arraylen >= 2 * MINLEN_BUFFER)) {
    if (adt_resizeArray(&(p->array), arraylen / 2)) {
      ERRMSG("cannot resize array");
      return ADT_ERROR;
    }
  }
  p->len--;
  return 0;
}


void *adt_indexList(adt_List *p, int index)
{
  ASSERT(p != NULL);
  if (index >= 0 && index < p->len) {
    return adt_indexArray(&(p->array), index);
  }
  else {
    BUG("index out of range");
  }
}


int adt_updateList(adt_List *p, int index, void *pelem)
{
  ASSERT(p != NULL);
  ASSERT(pelem != NULL);
  if (index >= 0 && index < p->len) {
    return adt_updateArray(&(p->array), index, pelem);
  }
  else {
    BUG("index out of range");
  }
}


int adt_resizeList(adt_List *p, int len)
{
  ASSERT(p != NULL);
  if (len < 0) {
    BUG("must have nonnegative length");
  }
  else if (len >= MINLEN_BUFFER) {
    if (adt_resizeArray(&(p->array), len)) {
      ERRMSG("cannot resize array");
      return ADT_ERROR;
    }
  }
  p->len = len;
  return 0;
}
