// The main program

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <iostream.h>
#include <iomanip.h>
#include "PDB.h"
#include "Parameters.h"
#include "Molecule.h"
#include "Vector.h"
#include "ComputeBonded.h"
#include "ComputeNonbonded.h"
#include "LoadDCD.h"
#include "Timestep.h"
#include "auxil.h"
#include "global.h"
#include "Energies.h"
#include "Selection.h"
#include "Slidewin.h" 
#include "MDEBase.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

#define VERSION 2.3

int main(int argc, char *argv[]) {
  cout << endl  
       << "   ******************************************************" << endl
       << "   **************         MDENERGY        ***************" << endl
       << "   ******               Version " << VERSION << "                ******" << endl
       << "   ******************************************************" << endl
       << endl;

  if (argc>25) usage_exit("", argv[0]);

  int i;
  

  MDEBase mde;

  mde.getarguments(argc, argv);

  if (mde.have_psf)
      cout << "MDENERGY> psf file                : " << mde.psfname << endl;
  cout << "MDENERGY> param file              : " << mde.parfile << endl;
  char ptype[7]; 
  switch(mde.paratype) {
      case CHARMM: strcpy(ptype, "CHARMM"); 
	  break;
      case AMBER:  strcpy(ptype, "AMBER"); 
	  break;
      case XPLOR:  strcpy(ptype, "XPLOR ");
	  break;
  }
  cout << "MDENERGY> param type              : " << ptype << endl;

  mde.init_outfiles();

  Molecule*   mol;

  // Initialize the parameters:
  if (mde.verbose) cerr<<"MDENERGY> Reading the parameters...\n";
  Parameters* params;
  if (mde.paratype == AMBER) { 
      // StringList *parmFilename = configList->find("parmfile");
      // StringList *coorFilename = configList->find("ambercoor");
      // "amber" is a temporary data structure, which records all
      // the data from the parm file. After copying them into
      // molecule, parameter and pdb structures, it will be deleted.
      Ambertoppar *amber;
      amber = new Ambertoppar;
      //if (amber->readparm(parmFilename->data))
      if (amber->readparm(mde.parfile)) {
	  params = new Parameters(amber, SCALE1_4FAC);
	  // for readExclusions ----v
	  mol = new Molecule(&mde.simparams, params, amber);
	  //if (coorFilename != NULL)
	  //pdb = new PDB(coorFilename->data, amber);
	  delete amber;
      }
      else
	  error_exit("Failed to read AMBER parm file!");
  }
  else {
      params = new Parameters(mde.parfile, mde.paratype);

      // Initialize the Molecule:
      cerr<<"MDENERGY> Building the molecule...\n";
      mol = new Molecule(&mde.simparams, params, mde.psfname);
  }
  mde.natoms=mol->numAtoms;

  cout << "MDENERGY> exclude                 : ";
  if (mde.simparams.exclude == EXCLUDE1_2) cout << "1-2" << endl;
  if (mde.simparams.exclude == EXCLUDE1_3) cout << "1-3" << endl;
  if (mde.simparams.exclude == EXCLUDE1_4) cout << "1-4" << endl;
  if (mde.simparams.exclude == EXCLUDE1_4SCALED) {
    cout << "scaled1-4" << endl;
    cout.setf(ios::fixed);  cout.precision(1);
    cout << "MDENERGY> scale14fac              : " << mde.simparams.scale14fac << endl;
    cout.precision(PRECISION); cout << resetiosflags (ios::floatfield);
  }
  
  // Fixed atoms
  // -----------
  
  // If there are fixed atoms, read indices, later intersect them with selections:
  if (mde.have_fixed) {
    mde.fixedatoms.read(mde.fixedatomsfile, mde.natoms);
    cout << "MDENERGY> Fixed atoms file        : " << mde.fixedatomsfile << endl;
    cout << "MDENERGY> Selection text          : " << mde.fixedatoms.seltext << endl;
    cout << "MDENERGY> Fixed atoms             : " << mde.fixedatoms.natoms  << endl;
    if (mde.verbose) mde.fixedatoms.checkprint();
 
    mol->set_fixed_atoms(mde.fixedatoms.atomsel, mde.fixedatoms.natoms);

    // Invert the fixed atoms list to get mobile atoms: 
    mde.fixedatoms.invert_list();
  }

  // Build all lists in Molecule:
  mol->build_lists_by_atom();  

  cout << "MDENERGY> nAtoms                  : " << mol->numAtoms << endl;
  cout << "MDENERGY> numBonds                : " << mol->numBonds << endl; 
  cout << "MDENERGY> numAngles               : " << mol->numAngles << endl; 
  cout << "MDENERGY> numDihedrals            : " << mol->numDihedrals << endl; 
  cout << "MDENERGY> numImpropers            : " << mol->numImpropers << endl; 
  cout << "MDENERGY> numHydrogenGroups       : " << mol->numHydrogenGroups << endl;
  cout << "MDENERGY> numExclusions (expl.)   : " << mol->numExclusions << endl;
  if (mde.verbose) cout << "MDENERGY> numTotalExclusions      : " << mol->numTotalExclusions << endl;

  if (mol->numExclusions) {
    cout << "MDENERGY> explicit exclusions     :" << endl;
    for (int i=0; i<mol->numExclusions; i++) {
      Exclusion* excl = mol->get_explicit_exclusion(i);
      cout << "                               " << excl->atom1 << "  " << excl->atom2 << endl;
    }
  }

  // H-bonds
  // --------

  if(mde.eflag & HBOND) {
      cerr<<"MDENERGY> Reading Hbond Parameters...\n";
      mde.build_hbonds(mol);
      if (mde.verbose) mde.print_donors(mol);
      if (mde.verbose) mde.print_acceptors(mol);
  }


  // Selection
  // ----------
  
  // If there is a selection, read it, else select all atoms:
  for (int s=0; s<2; s++) {
      if (s<mde.have_sel) {
	  mde.selection[s].read(mde.simparams.selname[s], mde.natoms);
	  cout << "MDENERGY> Selection file      : " << mde.simparams.selname[s] << endl;
	  cout << "MDENERGY> Selection text      : " << mde.selection[s].seltext << endl;
	  cout << "MDENERGY> Selected atoms      : " << mde.selection[s].natoms  << endl;
	  if (mde.have_fixed) {
	      // Intersect selection with fixed atoms:
	      mde.selection[s].intersect_with_fixed(&mde.fixedatoms);
	      cout << "MDENERGY> Selected nonfixed atoms : " << mde.selection[s].natoms  << endl;
	  }
	  mde.sel_natoms += mde.selection[s].natoms;
	  if (mde.verbose) mde.selection[s].checkprint();
      } else {
	  mde.selection[s].select_all(mde.natoms);
	  if (mde.have_sel<1 && s<1) {
	      mde.sel_natoms=mde.natoms;
	      cout << "MDENERGY> Selected atoms          : ALL" << endl;
	      if (mde.have_fixed) {
	  	  // Intersect selection with fixed atoms:
		  mde.selection[s].intersect_with_fixed(&mde.fixedatoms);
		  cout << "MDENERGY> Selected nonfixed atoms : " << mde.selection[s].natoms  << endl;
		  mde.sel_natoms=mde.selection[s].natoms;
	      }
	  }
      }
  }


  // Initialization
  // ---------------

  if (mde.verbose) cerr<<"MDENERGY> Initialize Bonded...\n";
  ComputeBonded    bonded(mol, params, mde.selection[0].natoms, mde.selection[0].byAtom, mde.eflag, mde.self, mde.have_fixed);
  
  if (mde.verbose) cerr<<"MDENERGY> Initialize Nonbonded...\n";
  ComputeNonbonded nonbonded(mol, params, mde.hbpar, &mde.simparams, mde.selection[0].byAtom, mde.have_sel);

     
  // Sliding window settings:
  // ------------------------
  // sliding window average over frames: width = 1+2*navg
  // (center element + navg values to the left and to the right)
  // to conserve the number of frames the first and last navg values
  // are filled with averages from smaller windows.
  
  // when buffer is full, a callback function specified by set_callback
  // is invoked, here:  write_avg_step(...);
  // The client data are supplied for the function write_avg_step(...)

  SlidingWindowAverage<float> slidewin_coor(mde.smoothing, 3*mde.sel_natoms);
  if (mde.smoothing) {
      Coor_clientdata coor_cl_data;
      coor_cl_data.dcd    = &mde.dcd_write;  // first time step
      coor_cl_data.fd     = mde.dcd_write.get_fd();  //coor_cl_data.nAtoms = sel_natoms;
      coor_cl_data.Natoms = mde.natoms;
      
      cout << "dcd.nAtoms=" << mde.dcd_write.num_atoms()<< endl;
      cout << mde.nframes << "   " << 3*mde.sel_natoms << "   " << mde.smoothing << endl;

      slidewin_coor.set_callback(&coor_cl_data, write_avg_coords);
      if (mde.ramping)      {slidewin_coor.set_ramping();  cout << "MDENERGY> Coor average tails    : ramped up/down\n";}
      else if (mde.padding) {slidewin_coor.set_padding();  cout << "MDENERGY> Coor average tails    : padded with first/last value\n";}
      else                  {slidewin_coor.set_cropping(); cout << "MDENERGY> Coor average tails    : cropped (results contain less frames)\n";}
  }                                                                            
                                                                                              
  SlidingWindowAverage<float> slidewin_energy(mde.navg, mde.sel_natoms);
  if (mde.have_efile && (mde.eflag & ENERGY) && mde.have_sel<2) {
      // Initialize clientdata for callback function
      E_clientdata energy_clientdata; 
      energy_clientdata.fd       = mde.E_file;
      energy_clientdata.Natoms   = mde.sel_natoms;
      energy_clientdata.logscale = mde.logarithmic;
      slidewin_energy.set_callback(&energy_clientdata, write_avg_energies);
      if (mde.ramping)      {slidewin_energy.set_ramping();  cout << "MDENERGY> E/atom average tails    : ramped up/down\n";}
      else if (mde.padding) {slidewin_energy.set_padding();  cout << "MDENERGY> E/atom average tails    : padded with first/last value\n";}
      else                  {slidewin_energy.set_cropping(); cout << "MDENERGY> E/atom average tails    : cropped (results contain less frames)\n";}
  }


  SlidingWindowAverage<Vector> slidewin_force(mde.navg, mde.sel_natoms);
  F_clientdata force_clientdata; 
  if (mde.have_ffile && (mde.eflag & FORCE) && mde.have_sel<2) {
      // Initialize clientdata for callback function
      force_clientdata.fd       = mde.F_file;
      force_clientdata.Natoms   = mde.sel_natoms;
      force_clientdata.total    = NULL;
      slidewin_force.set_callback(&force_clientdata, write_avg_forces);
      if (mde.ramping)      {slidewin_force.set_ramping();  cout << "MDENERGY> Force average tails    : ramped up/down\n";}
      else if (mde.padding) {slidewin_force.set_padding();  cout << "MDENERGY> Force average tails    : padded with first/last value\n";}
      else                  {slidewin_force.set_cropping(); cout << "MDENERGY> Force average tails    : cropped (results contain less frames)\n";}
  }

  SlidingWindowAverage<double> slidewin_E(mde.navg, NUME); 
  // Initialize clientdata for callback function
  Etot_clientdata etot_clientdata;
  etot_clientdata.fbeg  = mde.fbeg;
  etot_clientdata.eflag = mde.eflag;
  slidewin_E.set_callback(&etot_clientdata, print_energy_output);
  if (mde.avg_over_frames ||1) {
    if (mde.ramping)      {slidewin_E.set_ramping();  cout << "MDENERGY> Energy average tails    : ramped up/down\n";}
    else if (mde.padding) {slidewin_E.set_padding();  cout << "MDENERGY> Energy average tails    : padded with first/last value\n";}
    else                  {slidewin_E.set_cropping(); cout << "MDENERGY> Energy average tails    : cropped (results contain less frames)\n";}
  }
  double *E  = new double[NUME];
  memset((void*) E, 0, NUME*sizeof(double));


  EnergiesByAtom e(mol, mde.natoms, mde.sel_natoms, mde.selection[0].atomsel);
  ForcesByAtom   f(mol, mde.natoms, mde.sel_natoms, mde.selection[0].atomsel);
  Timestep *ts     = NULL;
  Timestep *vel_ts = NULL;

  Vector *pos     = new Vector[mde.natoms];
  Vector *vel     = new Vector[mde.natoms];
  float  *coorxyz = new float[3*mde.natoms];    // coordinate array for smoothing
  hbond  *hbondlist = NULL;
 
  // Initialize the Vectors
  memset((void*) pos,   0, mde.natoms*sizeof(Vector));
  memset((void*) vel,   0, mde.natoms*sizeof(Vector));
  

  if (mde.have_sel<2) {
    e.compute_dof_by_atom();
 
    if(mde.have_efile && mde.eflag & ENERGY) 
      write_E_file_header(mde.E_file, mde.selection[0].natoms, mde.selection[0].atomsel, 
			  mde.selection[0].seltext, mde.selected_energy, mde.navg, mde.self);
       
    if (mde.have_ffile && mde.eflag & FORCE) 
      write_F_file_header(mde.F_file, mde.selection[0].natoms, mde.selection[0].atomsel, 
			  mde.selection[0].seltext, mde.selected_energy, mde.navg, mde.self);

    if (mde.have_hbfile && mde.eflag & HBOND)
      write_HB_file_header(mde.HB_file, mde.selection[0].natoms, mde.selection[0].atomsel, 
			  mde.selection[0].seltext, mde.selected_energy, mde.self);
  }

  int numDegFreedom = 3*mde.natoms;
  // as the momentum is conserved throughout the simulation, the initial 
  // reduction of the degrees of freedom applies to every timestep.
  // That is the default setting for NAMD (COMmotion no)
  numDegFreedom -= 3;

  if (mde.numcoords) {
     cout << "MDENERGY> begin at frame          : " << mde.fbeg << endl;
     cout << "MDENERGY> stop at frame           : " << mde.fend << endl;
  }

  // Print a sample line of energy titles for mdfplot:
  cout << "ETITLE:"; print_energy_titles(mde.eflag);
  
  cerr<<"MDENERGY> Computing...\n";
  cout << endl;


  // ****************************************************** //
  // ****************     THE LOOP     ******************** //
  // ******        Loop over the dcd frames          ****** //
  // ******    (NAMD omits the frame at TS 0)        ****** //
  // ****************************************************** //

  int done_coordset = FALSE;
  int k=0;
  // Initialize the first coordinate set
  mde.init_coords();
  etot_clientdata.time  = mde.dcd_time;  // first time step
  etot_clientdata.dt    = mde.dcd_dt; // delta time step
  etot_clientdata.frame0 = 0-mde.navg;  // current frame number in this set

  while (mde.curcoord || mde.servermode) {
 
     // First process coordinate files if present:
     if (mde.curcoord) {
        if (mde.fend>=0 && k>mde.fend) break;
	
	if (done_coordset) {
	   mde.init_coords();
	   done_coordset=FALSE;
	   etot_clientdata.time  = mde.dcd_time;  // first time step
	   etot_clientdata.dt    = mde.dcd_dt;    // delta time step
	   etot_clientdata.frame0 = k-mde.navg;   // current frame number in this set
	}

	switch(mde.curcoord->type) {
	   case PDB_COOR: 
	      mde.curcoord=mde.curcoord->next;
	      done_coordset=TRUE;
	      if (k<mde.fbeg) {k++; continue;}
	      mde.pdb->get_all_positions(pos);
	      break;
	   case DCD_COOR: 
	      // Read the DCD coords and velocities
	      ts = mde.dcd->read();
	      if (ts==NULL) {
		 mde.curcoord=mde.curcoord->next;
		 done_coordset=TRUE;
		 continue;
	      }
	      if (k<mde.fbeg) {
		 k++; 
		 if (ts) delete ts;
		 ts = NULL;
		 continue;
	      }
	      ts->get_all_positions(pos); 

	      if (mde.numvel) { 
		 vel_ts = mde.veldcd->read();
		 if (vel_ts==NULL)
		    error_exit("ERROR: No more velDCD frames available. Check option -end.");
		 vel_ts->get_all_positions(vel); // velocity dcd has same format as coord dcd
	      }
	      break;
	   default: 
	      cerr << "No more coordinates available\n";
	      done_coordset=TRUE;
	      break;
	}
     }
     // If no coodinate files are present, then we are in server mode and look for data in the socket
     else {
       // if (poll_socket()) {
       // get_new_coordinates();
       //}
     }

	  
     Vector Fbond, Fangl, Fdihe, Fimpr, Fconf, Fvdw, Felec, Fnonb, Ftot;

     // Compute bonded forces and energies that are needed for analysis
     if(mde.eflag & EFCONF) {
       if(mde.eflag & EBOND || mde.eflag & FBOND) {
	 E[BOND]   = bonded.compute_bonds(pos, f.B, e.B, Fbond); 
         E[F_BOND] = Fbond.length();
	 force_clientdata.total = &Fbond;
       }
       if(mde.eflag & EANGL || mde.eflag & FANGL) {
	 E[ANGL]   = bonded.compute_angles(pos, f.A, e.A, Fangl);
	 E[F_ANGL] = Fangl.length();
	 force_clientdata.total = &Fangl;
        }
       if(mde.eflag & EDIHE || mde.eflag & FDIHE) {
	 E[DIHE]   = bonded.compute_dihedrals(pos, f.D, e.D, Fdihe);
	 E[F_DIHE] = Fdihe.length();
	 force_clientdata.total = &Fdihe;
        }
       if(mde.eflag & EIMPR || mde.eflag & FIMPR) {
	 E[IMPR]   = bonded.compute_impropers(pos, f.I, e.I, Fimpr);
	 E[F_IMPR] = Fimpr.length();
	 force_clientdata.total = &Fimpr;
       }

       if(mde.eflag & ECONF) E[CONF] = E[BOND] + E[ANGL] + E[DIHE] + E[IMPR]; 
       
       if(mde.eflag & FCONF) {
	 Fconf = Fbond + Fangl + Fdihe + Fimpr;
	 E[F_CONF] = Fconf.length();
       }
     }

     // Compute nonbonded forces and energies that are needed for analysis
     if((mde.eflag & ENONB) || (mde.eflag & HBOND) || (mde.eflag & FNONB)) {
       double Evdw, Eelec, Ehbon;
       nonbonded.request_energies(mde.eflag, mde.self, mde.verbose);
       nonbonded.compute(mol, pos, f.vdw, f.elec, e.vdw, e.elec, &hbondlist, Evdw, Eelec, Ehbon, Fvdw, Felec,
			 mde.selection[0].byAtom, mde.selection[1].byAtom, mde.have_sel);

       if(mde.eflag & EVDW)  E[VDW]  = Evdw;
       if(mde.eflag & EELEC) E[ELEC] = Eelec;
       if(mde.eflag & HBOND) E[HBON] = Ehbon;
       if(mde.eflag & ENONB) E[NONB] = Evdw + Eelec;

       if(mde.eflag & FVDW)  { E[F_VDW]  = Fvdw.length();  force_clientdata.total = &Fvdw; }
       if(mde.eflag & FELEC) { E[F_ELEC] = Felec.length(); force_clientdata.total = &Felec; }
       if(mde.eflag & FNONB) { Fnonb = Fvdw + Felec; E[F_NONB] = Fnonb.length(); }
     }


     if(mde.eflag & FTOT) { Ftot = Fconf + Fnonb; E[F_TOT] = Ftot.length(); }
     if(mde.eflag & EALL || (mde.eflag & EKIN)) E[TOT] = E[CONF] + E[NONB];

     if(mde.eflag & EKIN  && mde.have_sel<2) {
       E[KIN]  = e.compute_kinetic(vel);
       E[TEMP] = 2.0 * E[KIN] / ( numDegFreedom * BOLTZMAN );
       E[TOT] += E[KIN];
     }

     if (mde.have_sel<2) {
       e.compute_conf_and_total();
       f.compute_conf_and_total();
     }

     //cerr<<"k-fbeg="<<k-mde.fbeg<<endl;

     slidewin_E.add_data(k-mde.fbeg, E);

    double *e_ptr=NULL; // pointer to selected energy
    Vector *f_ptr=NULL; // pointer to selected force
    if (mde.eflag & EBOND)  e_ptr = e.B;
    if (mde.eflag & EANGL)  e_ptr = e.A;
    if (mde.eflag & EDIHE)  e_ptr = e.D;
    if (mde.eflag & EIMPR)  e_ptr = e.I;
    if (mde.eflag & EVDW)   e_ptr = e.vdw;
    if (mde.eflag & EELEC)  e_ptr = e.elec;
    if (mde.eflag == ENONB) e_ptr = e.nonb;
    if (mde.eflag & EKIN)   e_ptr = e.kin;
    if (mde.eflag == EALL)  e_ptr = e.tot;
    if (mde.eflag == ECONF) e_ptr = e.conf;
    if (mde.eflag & FBOND)  f_ptr = f.B; 
    if (mde.eflag & FANGL)  f_ptr = f.A;
    if (mde.eflag & FDIHE)  f_ptr = f.D;
    if (mde.eflag & FIMPR)  f_ptr = f.I;
    if (mde.eflag & FVDW)   f_ptr = f.vdw;
    if (mde.eflag & FELEC)  f_ptr = f.elec;
    if (mde.eflag == FCONF) {f_ptr = f.conf; force_clientdata.total = &Fconf; }
    if (mde.eflag == FNONB) {f_ptr = f.nonb; force_clientdata.total = &Fnonb; }
    if (mde.eflag == FTOT)  {f_ptr = f.tot;  force_clientdata.total = &Ftot; }

    if (mde.eflag & FORCE) {
      if (mde.eflag == FCONF || mde.eflag == FNONB || mde.eflag == FTOT) {
	for (i=0; i<mde.sel_natoms; i++) {
	  f.value[i] = (Vector) f_ptr[i];
	  //cerr << f.value[i] << endl;
	}
      } else {
	// Use only the selected atoms:
	for (i=0; i<mde.sel_natoms; i++) {
	  f.value[i] = f_ptr[mde.selection[0].atomsel[i]];
	  //cerr << f.value[i] << endl;
	}
      }
      slidewin_force.add_data(k, f.value);
    }

    if (mde.eflag & HBOND) {
      write_HB_step(mde.HB_file, nonbonded.get_numHbonds(), k, hbondlist);
    }

    // Pass the data to the sliding window and eventually write averages to the file:
    if (mde.have_efile && (mde.eflag & ENERGY) && mde.have_sel<2) {
      // Use only the selected atoms:
      for (int i=0; i<mde.sel_natoms; i++) {
	e.value[i] =  (float) e_ptr[mde.selection[0].atomsel[i]];
      }
      slidewin_energy.add_data(k, e.value);
    }

    if (mde.smoothing) {
        int i, j;
        for(i=0, j=0; i<mde.natoms; i++) {
            coorxyz[j++]=pos[i].x;
            coorxyz[j++]=pos[i].y;
            coorxyz[j++]=pos[i].z;
        }
        slidewin_coor.add_data(k, coorxyz);
    }
                                                                                       
    // Free the memory that was allocated in dcd.read();
    if (ts) delete ts;
    if (mde.numvel) delete vel_ts;
    
    //mde.dcd_time += mde.dcd_dt;
    //cerr <<"dcd_time="<<mde.dcd_time<<"   dcd_dt="<<mde.dcd_dt<<endl;
    k++;
  } // end frame cycle


  // Finish the averages and write remaining data
  slidewin_E.finish();
  if (mde.smoothing) 
      slidewin_coor.finish();
  if (mde.have_efile && (mde.eflag & ENERGY) && mde.have_sel<2) 
      slidewin_energy.finish();
  if (mde.have_ffile && (mde.eflag & FORCE)  && mde.have_sel<2) 
      slidewin_force.finish();
  

  if (mde.have_sel<2) {
    if (mde.have_efile && mde.eflag & ENERGY) close_write(mde.E_file);
    if (mde.have_ffile && mde.eflag & FORCE)  close_write(mde.F_file);
  }
  if (mde.eflag & HBOND) close_write(mde.HB_file);

  delete [] pos;
  delete [] vel;
  delete [] E;
  delete [] coorxyz;
  delete mol;
  delete params;


  cerr << "MDENERGY> Finished.\n";
  return 0;
}   
