#include "auxil.h"
#include "Energies.h"
#include "MDEBase.h"
#include "Parameters.h"
#include "Molecule.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

MDEBase::MDEBase() {
    eflag=0x0;
    have_hpar=0;
    patches=0;
    verbose=0;
    self=0;
    servermode=0;
    have_psf=0;
    have_par=0;
    have_amber=0;
    //have_vel=0;
    have_efile=0;
    have_ffile=0;
    have_hbfile=0;
    have_fixed=0;
    numcoords=0;
    numvel=0;
    avg_over_frames=0;
    navg=0;
    logarithmic=0;
    have_sel=0;
    natoms=0;
    sel_natoms=0; 
    vel_natoms=0;
    nframes=0;
    fbeg=0;
    fend=0;
    dcd_time=0;
    dcd_dt=0;
    smoothing=0;
    ramping=0;
    padding=0;
    selected_energy=NULL;
    fend_string = "last";
    E_file=NULL;
    F_file=NULL;
    HB_file=NULL;
    hparfile     = "/Projects/saam/mdenergy/hbonds/par_hbond.inp";
    paratype     = CHARMM;
    efile          = "efile";
    ffile          = "ffile";
    hbfile         = "hbfile";
    simparams.switchdist     = SWITCHDIST;
    simparams.cutoff         = CUTOFF;
    simparams.pairlistdist   = PAIRLISTDIST; 
    simparams.exclude        = EXCLUDE1_4;
    simparams.scale14fac     = SCALE1_4FAC;
    simparams.sncb           = 1/SCALE1_4FAC;
    simparams.readExclusions = 1;
    hbpar=NULL;
    pdb=NULL;
    dcd=NULL;
    veldcd=NULL;
    coordset=NULL;
    velset=NULL;
}

MDEBase::~MDEBase() {
    if (coordset) {
	struct Coordset *ptr;  //  Current position in list
	struct Coordset *next; //  Next position in list
	ptr=coordset;

	while (ptr != NULL)
	{
	    next=ptr->next;
	    delete ptr;
	    ptr=next;
	}
    }
    if (velset) {
	struct Coordset *ptr;  //  Current position in list
	struct Coordset *next; //  Next position in list
	ptr=velset;

	while (ptr != NULL)
	{
	    next=ptr->next;
	    delete ptr;
	    ptr=next;
	}
    }
    if (pdb) delete pdb;
    if (dcd) delete dcd;
    if (veldcd) delete veldcd;
    if (hbpar)  delete hbpar;
}

 // Go through the arguments
void MDEBase::getarguments(int argc, char** argv) {
    if (argc<=1) usage_exit("ERROR: No arguments specified!", "");

    curcoord = coordset;
    curvel   = velset;
    int ev = 1;
    while(ev < argc) 
    {
	// First frame to process
	if(!strcmp(argv[ev], "-beg")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		fbeg = atoi(argv[++ev]);     
	    }
	}
	// Last frame to process
	if(!strcmp(argv[ev], "-end")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		fend_string = argv[++ev];
	    }
	}
	// A pdb file
	if(!strcmp(argv[ev], "-pdb")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		add_to_coord_list(coordset, curcoord, argv[++ev], PDB_COOR);
		numcoords++;
	    }
	}
	// A dcd file
	if (!strcmp(argv[ev], "-dcd")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		add_to_coord_list(coordset, curcoord, argv[++ev], DCD_COOR);
		numcoords++;
	    }
	}
	// The psf file
	if(!strcmp(argv[ev], "-psf")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		psfname = argv[++ev]; 
		have_psf = 1;
	    }
	}
	// Parameter file
	if(!strcmp(argv[ev], "-par")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		parfile = argv[++ev]; 
		have_par = 1;
	    }
	}
	// Parameter file in XPLOR style
	if(!strcmp(argv[ev], "-xplor")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		parfile = argv[++ev]; 
		have_par = 1;
		paratype = XPLOR; 
	    }
	}
	// Parameter file in AMBER style (parm7 partop file)
	if(!strcmp(argv[ev], "-amber")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		parfile = argv[++ev]; 
		have_amber = 1;
		paratype = AMBER; 
	    }
	}

	// Run MDE in server mode
	if (!strcmp(argv[ev], "-server")) {
  	    servermode = 1;
	}

	// Don't read exclusions from parm file but autogenerate 
	// them in build_exclusions()
	if(!strcmp(argv[ev], "-noamberexcl")) {
	    simparams.readExclusions = 0; 
	}
	// Elstat 1-4 scaling factor in AMBER style
	if(!strcmp(argv[ev], "-sncb")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		simparams.sncb = atof(argv[++ev]); 
	    }
	}
	// Nonbonded exclusion policy
	if(!strncmp(argv[ev], "-excl", 4)) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		char *exclude = argv[++ev]; 
		if (!strcmp(exclude, "1-2")) simparams.exclude = EXCLUDE1_2;
		if (!strcmp(exclude, "1-3")) simparams.exclude = EXCLUDE1_3;
		if (!strcmp(exclude, "1-4")) simparams.exclude = EXCLUDE1_4;
		if (!strcmp(exclude, "scaled1-4")) simparams.exclude = EXCLUDE1_4SCALED;
	    }
	}
	// Elstat 1-4 scaling factor in CHARMM style
	if(!strcmp(argv[ev], "-scale14fac")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		simparams.scale14fac = atof(argv[++ev]); 
	    }
	}
	// hbond parameter file
	if(!strcmp(argv[ev], "-hpar")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		hparfile = argv[++ev]; 
		have_hpar = 1;
	    }
	}
	// Velocity dcd file (for kinetic energy calculation)
	if(!strcmp(argv[ev], "-vel")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		add_to_coord_list(velset, curvel, argv[++ev], DCD_COOR);
/*		struct Coordset *new_node = new Coordset;
		if (new_node == NULL) {
		    error_exit("memory allocation failed in MDEBase::getarguments\n");
		}
		new_node->source = argv[++ev];
		new_node->type = DCD_COOR;
		new_node->num=numvel;
		new_node->next = NULL;

		if (velset == NULL) { 
		    velset = new_node; 
		    curvel = new_node;
		}
		curvel->next = new_node;
		curvel = new_node;
*/	
		numvel++;
	    }
	}
	// Energy output file (energy for each atom, visualizable with vmd-script 'viz_energy.tcl')
	if(!strcmp(argv[ev], "-a") || !strcmp(argv[ev], "-atoms")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		efile = argv[++ev]; 
		have_efile=1;
	    }
	}
	// Force output file
	if(!strcmp(argv[ev], "-f")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		ffile = argv[++ev]; 
		have_ffile=1;
	    }
	}
	// HB_val output file for VMD
	if(!strcmp(argv[ev], "-hb")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		hbfile = argv[++ev]; 
		have_ffile=1;
	    }
	}
	
	// File with indices for an atom selection.
	// For Bonded interactions energies are always computed ONLY WITHIN this selection.
	// For Nonbonded energies the interaction of the selection with the 
	// ENVIRONMENT is included unless you additionally use the option -self.
	
	// If option -sel is used TWICE, the interaction energy between the 
	// two selections is computed (ONLY the interaction, atom by atom).
	if(!strcmp(argv[ev], "-sel")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		simparams.selname[have_sel] = argv[++ev]; 
		have_sel++;
	    }
	}
	// Compute ONLY SELF interaction for one selection
	if(!strcmp(argv[ev], "-self")) {
	    self=1;
	}

	// Provide indices of fixed atoms.
	if(!strcmp(argv[ev], "-fixed")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		fixedatomsfile = argv[++ev]; 
		have_fixed=1;
	    }
	}
	


	// Run Sliding window average of the energy over the frames: width = 1+2*navg
	// (center element + navg values to the left and to the right) 
	if(!strcmp(argv[ev], "-avg")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		navg = atoi(argv[++ev]); 
		avg_over_frames = 1;
		if (navg == 0) avg_over_frames = 0;
	    }
	}
	if(!strcmp(argv[ev], "-smooth")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		smoothing = atoi(argv[++ev]);
	    }
	}
	if(!strcmp(argv[ev], "-ramp")) {
	    ramping = 1; padding = 0;
	}
	if(!strcmp(argv[ev], "-pad")) {
	    padding = 1; ramping = 0;
	}
 	// switch distance for nonbonded interactions
	if(!strcmp(argv[ev], "-switch")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		simparams.switchdist = atoi(argv[++ev]); 
	    }
	}
	// switch distance for nonbondeed interactions
	if(!strcmp(argv[ev], "-cutoff")) {
	    if(argc > (ev + 1) && *(argv[ev+1]) != '-') {
		simparams.cutoff = atoi(argv[++ev]); 
		simparams.pairlistdist = simparams.cutoff;
	    }
	}
	// Write energy type from which to determine B-values
	// Only the last selected Energy will be written to the B-file
	if (!strncmp(argv[ev], "-bond",  5)) {eflag |= EBOND; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-angl",  5)) {eflag |= EANGL; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-dihe",  5)) {eflag |= EDIHE; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-impr",  5)) {eflag |= EIMPR; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-vdw",   4)) {eflag |= EVDW; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-elec",  5)) {eflag |= EELEC; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-nonb",  5)) {eflag |= ENONB; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-kin",   4)) {eflag |= EKIN; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-conf",  5)) {eflag |= ECONF; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-all",   4)) {eflag |= EALL; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-fbond", 6)) {eflag |= FBOND; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-fangl", 6)) {eflag |= FANGL; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-fdihe", 6)) {eflag |= FDIHE; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-fimpr", 6)) {eflag |= FIMPR; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-fconf", 6)) {eflag |= FCONF; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-ftot",  5)) {eflag |= FTOT; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-fvdw",  5)) {eflag |= FVDW; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-felec", 6)) {eflag |= FELEC; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-fnonb", 6)) {eflag |= FNONB; selected_energy=argv[ev];}
	if (!strncmp(argv[ev], "-hbon",  5)) {eflag |= HBOND; selected_energy=argv[ev];}
	
	// Write B-values on logarithmic scale
	//if(!strcmp(argv[ev], "-log")) {
	//    logarithmic = 1; 
	//}
	// verbose
	if(!strcmp(argv[ev], "-v")) {
	    verbose=1;
	}
	ev++;
    }

    // Remove the leading '-' from string
    trimfirst(selected_energy);

    // Rewind the linked list
    curcoord = coordset;
    curvel   = velset;

    if (!(have_par && have_psf) && !have_amber)
	usage_exit("ERROR: Need psf and par files or amber parm7 file! \n", argv[0]);
    if (!numcoords)
	usage_exit("ERROR: Need coordinates! (<pdb|dcd> files or IMD connection)\n", argv[0]);

    if (!strcmp(fend_string, "last")) {
	fend=-1;
	nframes = 0;
    } else {
	fend=atoi(fend_string);
	nframes = fend-fbeg+1;
	if (nframes<1) 
	    error_exit("ERROR: beg>end. Check options -beg and -end.");
    }

    if (nframes && nframes<=navg) 
	error_exit("ERROR: Number of frames must be greater than width of sliding window!");
    if (nframes && nframes<2*navg+1 && ramping)
	error_exit("ERROR: Number of frames must be greater than 1+2*width of sliding window (in ramping mode)!");

/*    if (numcoords>0 && (fbeg || fend)) {
	cout << "WARNING: Options beg/end ignored for multiple coordinate files!\n";
	fbeg=0; fend=0; fend_string="last";
    }
*/
    if (!eflag && !smoothing)
	bad_energy_select_exit();
    if (logarithmic)
	if (eflag==EVDW || eflag==EELEC)  
	    error_exit("ERROR: No logarithmic scale for vdw or elec available!");
    if (have_sel>2) 
	error_exit("ERROR: Maximum number of selections is 2!");
    if (have_sel==2 && !(eflag==EVDW ||eflag==EELEC ||eflag==ENONB || eflag & FNONB))
	error_exit("ERROR: Interaction energy can only be computed for -vdw, -elec or -nonb!");
    
    if (paratype==XPLOR) {
	if (!have_hpar) hparfile = "/Projects/saam/mdenergy/hbonds/par_hbond.xplor";
    }
} 
/*  End function getarguments  */

// Initialize the output files for writing
void MDEBase::init_outfiles() {
    if (have_sel<2) {
	if (have_efile && (eflag & ENERGY)) {
	    cout << "MDENERGY> efile      : " << efile << endl;
	    E_file  = open_write(efile);
	    if (E_file == NULL) error_exit("ERROR opening E_file\n");
	}
    }
    if (eflag & FORCE) {
	cout << "MDENERGY> ffile      : " << ffile << endl;
	F_file  = open_write(ffile);
	if (F_file == NULL) error_exit("ERROR opening F_file\n");
    }
    if (eflag & HBOND) {
	cout << "MDENERGY> hbfile     : " << hbfile << endl;
	HB_file = open_write(hbfile);
	if (HB_file == NULL) error_exit("ERROR opening HB_file\n");
    }
    
    if (smoothing) {
	int ret = dcd_write.openwrite("smoothed.dcd", natoms, 0, 1, 1, 1);
	if (!ret) error_exit("ERROR opening DCD file for writing");
    }

}

// Generate a new coordinate set from source and type and add it to the list
// I must pass the address of the pointers (&coord_set) in order to manipulate 
// the object it points to. 
void MDEBase::add_to_coord_list(struct Coordset* &coord_set, struct Coordset* &cur_coord,
				char* source, int type) {
    struct Coordset *new_node = new Coordset;
    if (new_node == NULL) {
	error_exit("memory allocation failed in MDEBase::getarguments\n");
    }

    new_node->source = source;
    new_node->type = type;
    new_node->num=numcoords; //FIXME: numvel for VELDCD
    new_node->next = NULL;
    
    if (coord_set == NULL) { 
	coord_set = new_node; 
	cur_coord = new_node;
    } else {
	cur_coord->next = new_node;
	cur_coord = new_node;
    }
    
    //cerr<<numcoords<<": root="<<coord_set<<"  cur="<<cur_coord<<"  next="<<cur_coord->next<<"  new_node="<<new_node<<"  source="<<source<<endl; 
}

void MDEBase::init_coords() {

    // Initialize the PDB data
    if (curcoord->type == PDB_COOR) {  
	cout << "MDENERGY> pdb file   : " << curcoord->source << endl;
	PDB* newpdb;
	newpdb = new PDB(curcoord->source);
	if (pdb) delete pdb;
	pdb = newpdb;

	if (!paratype==AMBER && curcoord==coordset && pdb->num_atoms()!=natoms)
	    error_exit("ERROR: Number of atoms changed between coordinate files!");
	//natoms = pdb->num_atoms();
//	dcd_time = 0;
	//cout << "MDENERGY> pdb frames : " << 1 << endl;
    }
    
    // Initialize the DCD data
    if (curcoord->type == DCD_COOR) {
	cout << "MDENERGY> dcd file   : " << curcoord->source << endl;
	DCD* newdcd;
	newdcd = new DCD;
	int err = newdcd->openread(curcoord->source); 
	if (err<=0) exit(0);
	if (dcd) delete dcd;
	dcd = newdcd;

	cout << "MDENERGY> dcd format : " << dcd->get_format() << endl;
	cout << "MDENERGY> dcd frames : " << dcd->num_frames() << endl;
	cout << "MDENERGY> delta t    : " << dcd->delta_t() << endl;
	cout << "MDENERGY> timestep   : " << dcd->timestep()*TIMEFACTOR << " fs" << endl;
	if (!paratype==AMBER && curcoord==coordset && dcd->num_atoms()!=natoms)
	    error_exit("ERROR: Number of atoms changed between coordinate files!");
	//natoms  = dcd->num_atoms();

	dcd_dt  = dcd->delta_t();
	dcd_time = dcd->start_t();
	//cout << "dcd_dt=" << dcd_dt<<"   dcd_time=" << dcd_time<<endl;
    }

    // Initialize the velocityDCD data
    if (numvel) {
	cout << "MDENERGY> veldcd file:" << curvel->source << endl;
	DCD* newveldcd;
	newveldcd = new DCD;
	int err = newveldcd->openread(curvel->source); 
	if (err<=0) exit(1);
	if (veldcd) delete veldcd;
	veldcd = newveldcd;
	vel_natoms = veldcd->num_atoms();

	if (vel_natoms != natoms)
	    error_exit("ERROR: Different number of atoms in coorfile and vel file");
	if (curcoord==coordset && veldcd->num_atoms()!=vel_natoms)
	    error_exit("ERROR: Number of atoms changed between coordinate files!");
	//if (vel_nframes != nframes)
	//  error_exit("ERROR: Different number of frames in coorfile and vel file");
    }
}

void MDEBase::build_hbonds(Molecule* mol) {
    // Read hbond parameter file:
    hbpar = new Parameters(hparfile, paratype);
    
    // Donor and acceptor lists are needed by ComputeNonbonded:
    cerr<<"MDENERGY> Building Hbond lists...\n";
    mol->build_donor_list(hbpar);
    mol->build_acceptor_list(hbpar);
}

void MDEBase::print_donors(Molecule* mol) {
    // Print a list of all donors:
    if (eflag & HBOND) {
	cout << "MDENERGY> Donors    : " << mol->numDonors << ":" << endl;
	cout << "MDENERGY> HDonors    : " << mol->numHDonors << ":" << endl;
	int i;
	for (i=0; i< mol->numHDonors; i++) {
	    int don = mol->get_donor(i);
	    int cnum, *cid;
	    mol->get_child_atoms(don, &cnum, &cid);
	    cerr << i << "  " << mol->get_resname(don)<<":"<<mol->get_resid(don) << "  " 
		 << mol->get_atomname(don);
	    cerr << " donor[" << mol->check_donor(don)
		 << "] hydrogen: " << mol->get_atomname(cid[0]) << endl;
	}
    }
}

void MDEBase::print_acceptors(Molecule* mol) {
    // Print a list of all acceptors:
    cout << "MDENERGY> Acceptors : " << mol->numAcceptors << ":" << endl;
    cout << "MDENERGY> HAcceptors : " << mol->numHAcceptors << ":" << endl;
    int i;
    for (i=0; i< mol->numHAcceptors; i++) {
	int acc = mol->get_acceptor(i);      // atom index by acceptorindex
	int nant, *ant;
	mol->get_antecedents(i, &nant, &ant);   // ant atom indexes by acc atom index

	cerr << i << "  " << mol->get_resname(acc)<<":"<<mol->get_resid(acc) << "  " << mol->get_atomname(acc);
	cerr << " acceptor[" << i << "] ante: ";
	if (nant==0) { cerr << "NONE" << endl; }
	else {
	    for (int j=0; j<nant; j++) {
		cerr << mol->get_atomname(*ant++) << " " << endl;
	    }
	}
    }
}
