#ifndef HBONDS_H
#define HBONDS_H

//#include "auxil.h" // error_exit()
#include "Mindy.h"
#include "Hydrogen.h"
#include <stdio.h>

#define UnknownAtom      0x00
#define HydrogenAtom     0x01
#define OxygenAtom       0x02

void error_exit(const char *s);


class DonAcc;

typedef struct patch {
    char pres[11];
    char segname1[11];
    char segname2[11];
    int resid1;
    int resid2;
} Patch; // this is for read_patches()


class HBonds {
    
    Atom *atoms;             //  Array of atom structures
    AtomNameInfo *atomNames; //  Array of atom name info
    Bond *bonds;             //  Array of bond structures

    Patch  *patches; 
    int *donor_type;            // index of the donor type
    int *acceptor_type;         // index of the acceptor type
    int *donorsByAtom;          // contains 1 for a donor, zero else.
    int *acceptorsByAtom;       // contains 1 for a donor, zero else
    int *antecedentsByAcceptor; // contains antecedent global index; indexed by Acceptor number
    int *patchlist;             // Array of indices of patched residues
    int *numBondsWithAtom;      //  Array of number of bonds each atom is involved in
    int **bondsWithAtom;	//  List of bonds involving each atom

 public:
    int numAtoms;         //  Number of atoms
    int numBonds;         //  Number of bonds
    int numDonors;          // Number of donors in psf file
    int numAcceptors;       // Number of acceptors in psf file
    int numHydrogenGroups;  //  Number of hydrogen groups
    int waterIndex;

    int numHDonors;         // Number of donors for build_donor_list()
    int numHAcceptors;      // Number of donors for build_acceptor_list()
    int numHAntecedents;    // Number of donors for build_antecedent_list()
    int numPatches;         // Number of patched residues (for build_donor_list())  

    HydrogenGroup hydrogenGroup; // indexes of "atoms" sorted by hydrogen groups

    HBonds();
    ~HBonds();
    void read_psf_file(const char* fname);
    void write_psf_file(const char* fname, const char* newfname, DonAcc *donacc);
    void read_atoms(FILE *fd);
    void read_bonds(FILE *fd);

    void read_donors(FILE *);     //  Read in hydrogen bond donors from .psf
    void read_acceptors(FILE *);  //  Read in hydrogen bond acceptors from .psf
    // analyze the atoms, and determine which are oxygen, etc.
    void build_atom_status(void);

    // Return the donor index if atom is H-bond donor (needs acceptorByAtom list build)
    // else return 0
    int check_donor(int atomnum) const { return donorsByAtom[atomnum]; };
    
    // Return acceptor index if atom is H-bond acceptor (needs acceptorByAtom list build)
    // else return 0
    int check_acceptor(int atomnum) const { return acceptorsByAtom[atomnum]; };

    // Return the antecedent of the acceptor with atom index atomnum
    int check_antecedent(int atomnum) const { 
	int acc=check_acceptor(atomnum);
	if (acc==-1) error_exit("ERROR: Antecedent for NON-acceptor requested in check_antecedent()!");
	return antecedentsByAcceptor[acc]; 
    };


    // Return the antecedent of the acceptor with atom index num
    int get_antecedent(int accnum) const { 
	if (accnum<0) error_exit("ERROR: Negative index in get_acceptor()!");
	if (accnum>=numHAcceptors) error_exit("ERROR: Index too high in get_acceptor()!");
	return antecedentsByAcceptor[accnum]; 
    };

    // Return the donor type index by atom number
    int get_donor_type(int atomnum) const { return donor_type[atomnum]; };

    // Return the acceptor type index by atom number
    int get_acceptor_type(int atomnum) const { return acceptor_type[atomnum]; };

    // Return all acceptors
    void get_acceptor_list(int **accept, int *num_acc);
    
    //Return all donors
    void get_donor_list(int **don, int *num_acc);
        
    void build_donor_list(int sel_n, DonAcc *donacc);
    void build_acceptor_list(int sel_n, DonAcc *donacc);
    void build_all_hbond_lists(const char* topname, int sel_n, int *selptr);
    void build_lists_by_atom();

    void read_patches(const char* patfile, const char * topfile, int sel_n, DonAcc *donacc); 
    int is_hydrogen(int);            // return true if atom is hydrogen
    int is_oxygen(int);              // return true if atom is oxygen

    //  Retrieve residue index
    int get_resid(int anum) const { return(atomNames[anum].resid); };
 
    //  Lookup atom id from segment, residue, and index in residue
    int *get_atoms_from_segid_and_residue(const char *segid, int resid, int *index) const;

    //  Retrieve atomname
    const char *get_atomname(int anum) const { 
	if (anum>=0) return(atomNames[anum].atomname);
	else error_exit("ERROR: Negative index in get_atomname()!");
	return NULL;
    };

    int  get_bond_atom1(int bnum) { return bonds[bnum].atom1; }
    int  get_bond_atom2(int bnum) { return bonds[bnum].atom2; }
    int  get_num_bonds_with_atom(int anum) { return numBondsWithAtom[anum]; }
    int* get_bonds_with_atom(int anum)     { return bondsWithAtom[anum]; }

    //  Retrieve residue name
    const char *get_resname(int anum) const { 
	if (anum>=0) return(atomNames[anum].resname); 
	else error_exit("ERROR: Negative index in get_resname()!");
	return NULL;
    };

    // Return number of child atoms of HydrogenGroup mother atom
    int get_num_child_atoms(int anum) {
	return hydrogenGroup[atoms[anum].hydrogenList].atomsInGroup-1;
    };

    void get_child_atoms(int anum, int* numChildren, int** cID);
};

#endif
