// Hbonds.C

#include <algorithm>
#include <iostream.h>
#include "HBonds.h"
#include "ReadDonAcc.h"
#include "strlib.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif

void error_exit(const char *s) {
  fprintf(stderr, "%s\n",s);
  exit(0);
}

HBonds::HBonds()
{
  patches=NULL;
  patchlist=NULL;

  atoms=NULL;
  atomNames=NULL;
  bonds=NULL;

 /*  Initialize counts to 0 */
  numDonors=0;
  numAcceptors=0;
  numAtoms=0;
  numBonds=0;
  numHDonors=0;
  numHAcceptors=0;
  numHAntecedents=0;
  numPatches=0;
}

HBonds::~HBonds()
{
    delete [] atoms;
    delete [] atomNames;
    delete [] bonds;

    delete [] patchlist;
    if (patches) delete [] patches;
}


void HBonds::read_psf_file(const char *fname)

{
  char err_msg[512];  //  Error message for error_exit
  char buffer[512];  //  Buffer for file reading
  int i;      //  Loop counter
  int NumTitle;    //  Number of Title lines in .psf file
  FILE *psf_file;    //  pointer to .psf file
  int ret_code;    //  ret_code from NAMD_read_line calls

  /* Try and open the .psf file for reading */
  if ( (psf_file = fopen(fname, "r")) == NULL)
  {
    sprintf(err_msg, "UNABLE TO OPEN .psf FILE %s", fname);
    error_exit(err_msg);
  }


  /*  Read till we have the first non-blank line of file    */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to see if we dropped out of the loop because of a     */
  /*  read error.  This shouldn't happen unless the file is empty */
  if (ret_code!=0)
  {
    sprintf(err_msg, "EMPTY .psf FILE %s", fname);
    error_exit(err_msg);
  }

  /*  The first non-blank line should contain the word "psf".    */
  /*  If we can't find it, die.               */
  if (!NAMD_find_word(buffer, "psf"))
  {
    sprintf(err_msg, "UNABLE TO FIND \"PSF\" STRING IN PSF FILE %s",
       fname);
    error_exit(err_msg);
  }

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to see if we dropped out of the loop because of a     */
  /*  read error.  This shouldn't happen unless there is nothing  */
  /*  but the PSF line in the file        */
  if (ret_code!=0)
  {
    sprintf(err_msg, "MISSING EVERYTHING BUT PSF FROM %s", fname);
    error_exit(err_msg);
  }

  /*  This line should have the word "NTITLE" in it specifying    */
  /*  how many title lines there are        */
  if (!NAMD_find_word(buffer, "NTITLE"))
  {
    sprintf(err_msg,"CAN NOT FIND \"NTITLE\" STRING IN PSF FILE %s",
       fname);
    error_exit(err_msg);
  }

  sscanf(buffer, "%d", &NumTitle);

  /*  Now skip the next NTITLE non-blank lines and then read in the*/
  /*  line which should contain NATOM        */
  i=0;

  while ( ((ret_code=NAMD_read_line(psf_file, buffer)) == 0) && 
    (i<NumTitle) )
  {
    if (!NAMD_blank_string(buffer))
      i++;
  }

  /*  Make sure we didn't exit because of a read error    */
  if (ret_code!=0)
  {
    sprintf(err_msg, "FOUND EOF INSTEAD OF NATOM IN PSF FILE %s", 
       fname);
    error_exit(err_msg);
  }

  while (NAMD_blank_string(buffer))
  {
    NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we have the line we want      */
  if (!NAMD_find_word(buffer, "NATOM"))
  {
    sprintf(err_msg, "DIDN'T FIND \"NATOM\" IN PSF FILE %s",
       fname);
    error_exit(err_msg);
  }

  /*  Read in the number of atoms, and then the atoms themselves  */
  sscanf(buffer, "%d", &numAtoms);

  read_atoms(psf_file);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    error_exit("EOF ENCOUNTERED LOOKING FOR NBONDS IN PSF");
  }

  /*  Look for the string "NBOND"          */
  if (!NAMD_find_word(buffer, "NBOND"))
  {
    error_exit("DID NOT FIND NBOND AFTER ATOM LIST IN PSF");
  }

  /*  Read in the number of bonds and then the bonds themselves  */
  sscanf(buffer, "%d", &numBonds);

  if (numBonds)
    read_bonds(psf_file);


  while ( (ret_code==0) && (!NAMD_find_word(buffer, "NDON")))
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    error_exit("EOF ENCOUNTERED LOOKING FOR NDON IN PSF");
  }

  /*  Look for the string "NDON"        */
  if (!NAMD_find_word(buffer, "NDON"))
  {
    error_exit("DID NOT FIND NDON AFTER ATOM LIST IN PSF");
  }

  /*  Read in the number of hydrogen bond donors and then the donors */
  sscanf(buffer, "%d", &numDonors);

  if (numDonors)
    read_donors(psf_file);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    error_exit("EOF ENCOUNTERED LOOKING FOR NACC IN PSF");
  }

  /*  Look for the string "NACC"        */
  if (!NAMD_find_word(buffer, "NACC"))
  {
    error_exit("DID NOT FIND NACC AFTER ATOM LIST IN PSF");
  }

  /*  Read in the number of hydrogen bond donors and then the donors */
  sscanf(buffer, "%d", &numAcceptors);

  if (numAcceptors)
    read_acceptors(psf_file);


  /*  Close the .psf file.  There is a Group section in the .psf  */
  /*  file after the NNB section, but currently, NAMD does not    */
  /*  use this section for anything.        */
  fclose(psf_file);

  //  analyze the data and find the status of each atom
  //build_atom_status();

  return;
}
/*      END OF FUNCTION read_psf_file      */


void HBonds::write_psf_file(const char *fname, const char *newfname, DonAcc *donacc)

{
  char err_msg[512];  //  Error message for error_exit
  char buffer[512];   //  Buffer for file reading
  FILE *psf_read;     //  pointer to .psf file
  FILE *psf_write;    //  pointer to .psf file
  int ret_code;       //  ret_code from NAMD_read_line calls
  long file_pos, file_pos2;      //  File position

  /* Try and open the .psf file for reading */
  if ( (psf_read = fopen(fname, "r")) == NULL)
  {
    sprintf(err_msg, "UNABLE TO OPEN .psf FILE %s", fname);
    error_exit(err_msg);
  }

  /* Try and open the .psf file for writing */
  if ( (psf_write = fopen(newfname, "w+")) == NULL)
  {
    sprintf(err_msg, "UNABLE TO OPEN .psf FILE %s FOR WRITING", newfname);
    error_exit(err_msg);
  }

  /*  Loop and read lines until NDON occurs.      */
  do {
      /*  Get the line from the file        */
      ret_code = NAMD_read_line_keepwhite(psf_read, buffer);
      file_pos = ftell(psf_write);

      /* Copy the line to the new PSF file */
      fprintf(psf_write, "%s\n", buffer);

      /*  If its blank or a comment, skip it      */
      if ( (NAMD_blank_string(buffer)) || (buffer[0] == '!') )
	  continue;
      
      /*  Check to make sure we didn't hit the EOF      */
      if (ret_code != 0)
          error_exit("EOF ENCOUNTERED LOOKING FOR NDON IN PSF");

      if (ferror(psf_write)) 
          error_exit("ERROR WRITING DONOR/ACCEPTOR INFO TO PSF FILE!");
      
      /*  Look for the string "NDON"          */
      if (NAMD_find_word(buffer, "NDON"))
      {
	  int npairs=0;
	  numHDonors=0;
          // Write a dummy for the real NACC line
	  fprintf(psf_write, "        \n");	  

	  int anum;
	  for (anum=0; anum<numAtoms; anum++) {
	      const char *res = get_resname(anum);
	      const char *don = get_atomname(anum);
	      if (donacc->is_donor(res, don)) {
		  int j, numH;
		  int *cID;
		  get_child_atoms(anum, &numH, &cID);
		  for (j=0 ; j<numH; j++) {
		      if (donacc->is_donor_H(res, don, get_atomname(cID[j]))) {
			  fprintf(psf_write, "%-8i%-8i", anum+1, cID[j]+1);
			  numHDonors++;
			  npairs++;
			  if (npairs >= 4) {
			      fprintf(psf_write, "\n");
			      npairs = 0;
			  }      
		      }
		  }
	      }
	  }
	  int countDon=0;
	  for (int p=0; p<numPatches; p++) {
	      int numARes; // number of Atoms in current patched residue
	      cout << patches[p].pres << " " << patches[p].segname1 << ':' << patches[p].resid1
		   << " " << patches[p].segname2 << ":" << patches[p].resid2 << endl;
	      // Get all indexes for the current patched residue
	      int *index = get_atoms_from_segid_and_residue(patches[p].segname1, patches[p].resid1, &numARes);
	      
	      // Loop over all atoms in current patched residue
	      for (int k=0; k<numARes; k++) {
		  int anum = index[k];
		  
		  // Check if current atom is a DONOR (according to definition in PRES)
		  if (donacc->is_donor(patches[p].pres, get_atomname(anum))) {
		      int j, numH;
		      int *cID;
		      get_child_atoms(anum, &numH, &cID);
		      for (j=0 ; j<numH; j++) {
			  if (donacc->is_donor_H(patches[p].pres, get_atomname(anum), get_atomname(cID[j]))) {
			      fprintf(psf_write, "%-8i%-8i", anum+1, cID[j]+1);
			      numHDonors++;
			      npairs++;
			      if (npairs >= 4) {
				  fprintf(psf_write, "\n");
				  npairs = 0;
			      }      
			      cerr<<anum<<") pres: "<<patches[p].pres<<" res: "<<get_resname(anum)<<":"<<get_resid(anum)
				  <<"  donname: "<<get_atomname(anum)<<"  hyd: "<<get_atomname(cID[j])<<endl;
			  
			      countDon++;
			  }
		      }
		  }
	      }
	  }

	  // Rewind to NDON line and write numHDonors:
	  file_pos2 = ftell(psf_write);
	  fseek(psf_write, file_pos, SEEK_SET);
	  fprintf(psf_write, "%d !NDON: donors", numHDonors);	  
	  fseek(psf_write, file_pos2, SEEK_SET);

	  cout << "numHDonors = " << numHDonors <<endl;
      }
      /*  Look for the string "NACC"          */
      if (NAMD_find_word(buffer, "NACC"))
      {
	  numHAcceptors=0;
	  int npairs=0;
          // Write a dummy for the real NACC line
	  fprintf(psf_write, "        \n");	  

	  int anum;
	  for (anum=0; anum<numAtoms; anum++) {
	      const char *res = get_resname(anum);
	      const char *acc = get_atomname(anum);
	      if (donacc->is_acceptor(res, acc)) {
		  int nBonds = get_num_bonds_with_atom(anum);
		  int *antelist = new int[nBonds]; // WHY -1 ???
		  int *bnum = get_bonds_with_atom(anum);
		  int j;
		  for (j=0; j<nBonds; j++) {
		      int atom1 = get_bond_atom1(bnum[j]);
		      int atom2 = get_bond_atom2(bnum[j]);
		      if (atom1==anum) antelist[j]=atom2;
		      else if (atom2==anum) antelist[j]=atom1;
		  }
		  for (j=0; j<nBonds; j++) {
		      fprintf(psf_write, "%-8i%-8i", anum+1, antelist[j]+1);
		      npairs++; 
		      numHAcceptors++;
		      if (npairs >= 4) {
			  fprintf(psf_write, "\n");
			  npairs = 0;
		      }
		  }
	      }
	  }
	  int countAcc=0;
	  for (int p=0; p<numPatches; p++) {
	      int numARes; // number of Atoms in current patched residue
	      cout << patches[p].pres << " " << patches[p].segname1 << ':' << patches[p].resid1
		   << " " << patches[p].segname2 << ":" << patches[p].resid2 << endl;
	      // Get all indexes for the current patched residue
	      int *index = get_atoms_from_segid_and_residue(patches[p].segname1, patches[p].resid1, &numARes);
	      
	      // Loop over all atoms in current patched residue
	      for (int k=0; k<numARes; k++) {
		  int anum = index[k];
		  
		  // Check if current atom is an ACCEPTOR (according to definition in PRES)
		  if (donacc->is_acceptor(patches[p].pres, get_atomname(anum))) {
		      int nBonds = get_num_bonds_with_atom(anum);
		      int *antelist = new int[nBonds]; // WHY -1 ???
		      int *bnum = get_bonds_with_atom(anum);
		      int j;
		      for (j=0 ; j<nBonds; j++) {
			  int atom1 = get_bond_atom1(bnum[j]);
			  int atom2 = get_bond_atom2(bnum[j]);
			  if (atom1==anum) antelist[j]=atom2;
			  else if (atom2==anum) antelist[j]=atom1;
		      }
		      for (j=0; j<nBonds; j++) {
			  if (donacc->is_acceptor_A(patches[p].pres, get_atomname(anum), get_atomname(antelist[j]))) {
			      fprintf(psf_write, "%-8i%-8i", anum+1, antelist[j]+1);
			      npairs++; 
			      numHAcceptors++;
			      if (npairs >= 4) {
				  fprintf(psf_write, "\n");
				  npairs = 0;
			      }
			      cerr<<anum<<") pres: "<<patches[p].pres<<" res: "<<get_resname(anum)<<":"<<get_resid(anum)
				  <<"  donname: "<<get_atomname(anum)<<"  hyd: "<<get_atomname(antelist[j])<<endl;
			  
			      countAcc++;
			  }
		      }
		  }
	      }
	  }	  

	  // Rewind to NDON line and write numHDonors:
	  file_pos2 = ftell(psf_write);
	  fseek(psf_write, file_pos, SEEK_SET);
	  fprintf(psf_write, "%d !NACC: acceptors", numHAcceptors);	  
	  fseek(psf_write, file_pos2, SEEK_SET);

	  cout << "numHAcceptors = " << numHAcceptors << endl;
      }
  } while (ret_code == 0);

  fclose(psf_read);
  fclose(psf_write);

  return;
}
/* END OF FUNCTION write_psf_file */



/************************************************************************/
/*                  */
/*        FUNCTION read_atoms      */
/*                  */
/*   INPUTS:                */
/*  fd - file pointer to the .psf file        */
/*  params - Parameters object to use for parameters    */
/*                  */
/*  this function reads in the Atoms section of the .psf file.      */
/*   This section consists of numAtoms lines that are of the form:  */
/*     <atom#> <mol> <seg#> <res> <atomname> <atomtype> <charge> <mass> */
/*   Each line is read into the appropriate entry in the atoms array.   */
/*   The parameters object is then used to determine the vdW constants  */
/*   for this atom.              */
/*                  */
/************************************************************************/

void HBonds::read_atoms(FILE *fd)

{

  char buffer[512];  // Buffer for reading from file
  int atom_number=0;  // Atom number 
  int last_atom_number=0; // Last atom number, used to assure
        // atoms are in order
  char segment_name[11];   // Segment name
  char residue_number[11]; // Residue number
  char residue_name[11];   // Residue name
  char atom_name[11];      // Atom name
  char atom_type[11];      // Atom type
  double charge;           // Charge for the current atom
  double mass;             // Mass for the current atom
  int read_count;          // Number of fields read by sscanf

  /*  Allocate the atom arrays          */
  atoms     = new Atom[numAtoms];
  atomNames = new AtomNameInfo[numAtoms];
  hydrogenGroup.resize(0);

  if (atoms == NULL || atomNames == NULL )
  {
    error_exit("memory allocation failed in Molecule::read_atoms");
  }

  /*  Loop and read in numAtoms atom lines.      */
  while (atom_number < numAtoms)
  {
      /*  Get the line from the file        */
      NAMD_read_line(fd, buffer);
      
      /*  If its blank or a comment, skip it      */
      if ( (NAMD_blank_string(buffer)) || (buffer[0] == '!') )
	  continue;
      
      /*  Parse up the line          */
      read_count=sscanf(buffer, "%d %s %s %s %s %s %lf %lf",
			&atom_number, segment_name, residue_number,
			residue_name, atom_name, atom_type, &charge, &mass);
      
      /*  Check to make sure we found what we were expecting  */
      if (read_count != 8)
      {
	  char err_msg[128];
	  
	  sprintf(err_msg, "BAD ATOM LINE FORMAT IN PSF FILE IN ATOM LINE %d\nLINE=%s",
		  last_atom_number+1, buffer);
	  error_exit(err_msg);
      }
      
      /*  Make sure the atoms were in sequence    */
      if (atom_number != last_atom_number+1)
      {
	  char err_msg[128];
	  
	  sprintf(err_msg, "ATOM NUMBERS OUT OF ORDER AT ATOM #%d OF PSF FILE",
		  last_atom_number+1);
	  error_exit(err_msg);
      }
      
      last_atom_number++;
      
      /*  Dynamically allocate strings for atom name, atom    */
      /*  type, etc so that we only allocate as much space    */
      /*  for these strings as we really need      */
      int seglength = strlen(segment_name)+1;
      int reslength = strlen(residue_name)+1;
      int namelength = strlen(atom_name)+1;
      int typelength = strlen(atom_type)+1;
      
      atomNames[atom_number-1].resid = atoi(residue_number);
      atomNames[atom_number-1].resname = new char[reslength];
      atomNames[atom_number-1].segname = new char[seglength];
      atomNames[atom_number-1].atomname = new char[namelength];
      atomNames[atom_number-1].atomtype = new char[typelength];
      
      if (atomNames[atom_number-1].resname == NULL)
	  error_exit("memory allocation failed in Molecule::read_atoms");
      if (atomNames[atom_number-1].segname == NULL)
	  error_exit("memory allocation failed in Molecule::read_atoms");
      
      /*  Put the values from this atom into the atoms array  */
      strcpy(atomNames[atom_number-1].segname, segment_name);
      strcpy(atomNames[atom_number-1].resname, residue_name);
      strcpy(atomNames[atom_number-1].atomname, atom_name);
      strcpy(atomNames[atom_number-1].atomtype, atom_type);
      atoms[atom_number-1].mass = mass;
      atoms[atom_number-1].charge = charge;
      atoms[atom_number-1].status = UnknownAtom;
      
      /*  Determine the type of the atom (H or O) */
      if (atoms[atom_number-1].mass <=3.5) {
	  atoms[atom_number-1].status |= HydrogenAtom;
      } else if ((atomNames[atom_number-1].atomname[0] == 'O') && 
		 (atoms[atom_number-1].mass >= 14.0) && 
		 (atoms[atom_number-1].mass <= 18.0)) {
	  atoms[atom_number-1].status |= OxygenAtom;
      }
  }
  
  return;
}
/*      END OF FUNCTION read_atoms      */
  
  
  
/************************************************************************/
/*                  */
/*      FUNCTION read_bonds        */
/*                  */
/*  read_bonds reads in the bond section of the .psf file.  This    */
/*  section contains a list of pairs of numbers where each pair is      */
/*  represents two atoms that are bonded together.  Each atom pair is   */
/*  read in.  Then that parameter object is queried to determine the    */
/*  force constant and rest distance for the bond.      */
/*                  */
/************************************************************************/

void HBonds::read_bonds(FILE *fd)

{
  int atom_nums[2];  // Atom indexes for the bonded atoms
  char atom1name[11];  // Atom type for atom #1
  char atom2name[11];  // Atom type for atom #2
  register int j;      // Loop counter
  int num_read=0;    // Number of bonds read so far
  int origNumBonds = numBonds;   // number of bonds in file header

  /*  Allocate the array to hold the bonds      */
  bonds=new Bond[numBonds];

  if (bonds == NULL)
  {
    error_exit("memory allocations failed in Molecule::read_bonds");
  }

  /*  Loop through and read in all the bonds      */
  while (num_read < numBonds)
  {
    /*  Loop and read in the two atom indexes    */
    for (j=0; j<2; j++)
    {
      /*  Read the atom number from the file.         */
      /*  Subtract 1 to convert the index from the    */
      /*  1 to NumAtoms used in the file to the       */
      /*  0 to NumAtoms-1 that we need    */
      atom_nums[j]=NAMD_read_int(fd, "BONDS")-1;

      /*  Check to make sure the index isn't too big  */
      if (atom_nums[j] >= numAtoms)
      {
        char err_msg[128];

        sprintf(err_msg, "BOND INDEX %d GREATER THAN NATOM %d IN BOND # %d IN PSF FILE", atom_nums[j]+1, numAtoms, num_read+1);
        error_exit(err_msg);
      }
    }

    /*  Get the atom type for the two atoms.  When we query */
    /*  the parameter object, we need to send the atom type */
    /*  that is alphabetically first as atom 1.    */
    if (strcasecmp(atomNames[atom_nums[0]].atomtype, 
         atomNames[atom_nums[1]].atomtype) < 0)
    {
      strcpy(atom1name, atomNames[atom_nums[0]].atomtype);
      strcpy(atom2name, atomNames[atom_nums[1]].atomtype);
    }
    else
    {
      strcpy(atom2name, atomNames[atom_nums[0]].atomtype);
      strcpy(atom1name, atomNames[atom_nums[1]].atomtype);
    }

    /*  Query the parameter object for the constants for    */
    /*  this bond            */
    Bond *b = &(bonds[num_read]);
    //params->assign_bond_index(atom1name, atom2name, b);

    /*  Assign the atom indexes to the array element  */
    b->atom1=atom_nums[0];
    b->atom2=atom_nums[1];

    /*  Make sure this isn't a fake bond meant for shake in x-plor.  */
    //double k, x0;
    //params->get_bond_params(&k,&x0,b->bond_type);
    //if ( k == 0. ) --numBonds;  // fake bond
    //else ++num_read;  // real bond
    ++num_read;
  }

  /*  Tell user about our subterfuge  */
  if ( numBonds != origNumBonds ) {
    cout <<   "Ignored " << origNumBonds - numBonds <<
            " bonds with zero force constants.\n" << endl;
    cout <<  
	"Will get H-H distance in rigid H2O from H-O-H angle.\n" << endl;
  }

  return;
}
/*      END OF FUNCTION read_bonds      */



// go through the molecular structure, analyze the status of each atom,
// and save the data in the Atom structures stored for each atom.  This
// could be built up incrementally while the molecule is being read in,
// but doing it all in one shot allows us to just send the basic info
// over the network and have each node calculate the rest of the data on
// it's own.
void HBonds::build_atom_status(void) {
  register int i;
  int a1, a2;
  
  // initialize information for each atom (note that the status has
  // already been initialized during the read/receive phase)
  HydrogenGroupID *hg;
  hg = new HydrogenGroupID[numAtoms];
  for (i=0; i < numAtoms; i++) {
    hg[i].atomID = i;  // currently unsorted
    hg[i].atomsInGroup = 1;  // currently only 1 in group
    hg[i].isGP = 1;  // assume it is a group parent
    hg[i].GPID = i;  // assume it is a group parent
    hg[i].sortVal = 0;  // for group sorting
    int j=0;
    while(j<4) hg[i].childID[j++]=-1;
  }
  
  // find which atom each hydrogen is bound to
  // also determine number of atoms in each group
  for (i=0; i < numBonds; i++) {
    a1 = bonds[i].atom1;
    a2 = bonds[i].atom2;
    if (is_hydrogen(a1))
      {
	// check for hydrogen gas...  For H2, explicitly define the group parent.
	// I have been informed that H3 is not a concern.  This is good since
	// this code will fail for H3.
	if (is_hydrogen(a2)) {
	  hg[a1].isGP = 1;
	  cout <<   "Found H-H bond - are you sure?" << endl;
	} else {
	  hg[a2].childID[hg[a2].atomsInGroup-1] = a1; // add H-atom index to childID list
	  hg[a2].atomsInGroup++;
	  hg[a1].atomsInGroup = 0;
	  hg[a1].GPID = a2;
	  hg[a1].isGP = 0;
	  // check for waters (put them in their own groups: OH or OHH)
	  if (is_oxygen(a2))  hg[a2].sortVal++;
	}
      }
    if (is_hydrogen(a2))
      {
	hg[a1].childID[hg[a1].atomsInGroup-1] = a2; // add H-atom index to childID list
	hg[a1].atomsInGroup++;
	hg[a2].atomsInGroup = 0;
	hg[a2].GPID = a1;
	hg[a2].isGP = 0;
	// check for waters (put them in their own groups: OH or OHH)
	if (is_oxygen(a1))  hg[a1].sortVal++;
      }
  }
  
  // sort the hydrogenGroup list and count number of groups
  numHydrogenGroups = 0;
  for(i=0; i<numAtoms; i++)
  {
    // make H follow their group parents.
    if (!hg[i].isGP)  hg[i].sortVal = hg[hg[i].GPID].sortVal;
    else ++numHydrogenGroups;
    // add to list to sort
    hydrogenGroup.push_back(hg[i]);
  }
  sort(hydrogenGroup.begin(), hydrogenGroup.end());

  // finally, add the indexing from atoms[] to hydrogenGroup[]
  waterIndex = numAtoms;
  for(i=0; i<numAtoms; i++) {
    atoms[hydrogenGroup[i].atomID].hydrogenList = i;
    // identify where waters start
    if ((hydrogenGroup[i].sortVal == 2) && (i < numAtoms))
      waterIndex = i;
  }
  delete [] hg;
}

int HBonds::is_hydrogen(int anum) {
  return ((atoms[anum].status & HydrogenAtom) != 0);
}

int HBonds::is_oxygen(int anum) {
  return ((atoms[anum].status & OxygenAtom) != 0);
}


// Read the donors, and ignore them.
void HBonds::read_donors(FILE *fd) {
  for (int i=0; i<numDonors; i++) {
    NAMD_read_int(fd, "DONORS"); // -1 ?
    NAMD_read_int(fd, "DONORS"); // -1 ?
  }
}

void HBonds::read_acceptors(FILE *fd) {
  for (int i=0; i<numAcceptors; i++) {
    NAMD_read_int(fd, "ACCEPTORS"); //-1;
    NAMD_read_int(fd, "ACCEPTORS"); //-1;
  }
}

    /************************************************************************/
    /*                  */
    /*      FUNCTION build_lists_by_atom      */
    /*                  */
    /*  This function builds O(NumAtoms) arrays that store the bonds,   */
    /*  angles, dihedrals, and impropers, that each atom is involved in.    */
    /*  This is a space hog, but VERY fast.  This will certainly have to    */
    /*  change to make things scalable in memory, but for now, speed is the */
    /*  thing!                */
    /*                  */
    /************************************************************************/

void HBonds::build_lists_by_atom()
    
{
    register int i;      //  Loop counter
    
    bondsWithAtom = new int *[numAtoms];
    
    numBondsWithAtom = new int[numAtoms];
    memset((void*) numBondsWithAtom, 0, numAtoms*sizeof(int));
    
    int *byAtomSize = new int[numAtoms];
    
    //  Build the bond lists
    for (i=0; i<numAtoms; i++)
    {
	byAtomSize[i] = 0;
    }
    for (i=0; i<numBonds; i++)
    {
	byAtomSize[bonds[i].atom1]++;
	byAtomSize[bonds[i].atom2]++;
	// count the number of bonds atom is involved in
	numBondsWithAtom[bonds[i].atom1]++;
	numBondsWithAtom[bonds[i].atom2]++;
    }
    for (i=0; i<numAtoms; i++)
    {
	bondsWithAtom[i] = new int[byAtomSize[i]+1];
	bondsWithAtom[i][byAtomSize[i]] = -1;
	byAtomSize[i] = 0;
    }
    for (i=0; i<numBonds; i++)
    {
	int a1 = bonds[i].atom1;
	int a2 = bonds[i].atom2;
	bondsWithAtom[a1][byAtomSize[a1]++] = i;
	bondsWithAtom[a2][byAtomSize[a2]++] = i;
       }
    delete [] byAtomSize;
}


void HBonds::get_child_atoms(int anum, int* numChildren, int** cID) {
  *numChildren = hydrogenGroup[atoms[anum].hydrogenList].atomsInGroup-1;
  *cID = hydrogenGroup[atoms[anum].hydrogenList].childID;
}

void HBonds::read_patches(const char* patfile, const char* topfile, int sel_n, DonAcc *donacc)
{
  int len = 0;
  char tmp[1000];
  char keyword[100];
  char pres[11];
  char *delim;
  const char *resid;
  char res1[20];
  char res2[20];

  memset((void*) keyword, 0, 100*sizeof(int));

  FILE* fd = fopen(patfile, "r");
  if (fd==NULL) {
    cerr << "ERROR: Couldn't open file " << patfile << endl;
    exit(1);
  }
  // Count the patches
  while (!feof(fd)) {
    fgets(tmp, 1000, fd);        // read the line
    strcpy(keyword, "\0");       // Reset ketword
    sscanf (tmp, "%s", keyword); // get the first token
    //cerr<<len<<". "<<keyword<<endl;
    if (!strcmp(keyword, "patch")|| !strcmp(keyword, "PATCH")) len++;
    if (ferror(fd)) error_exit("ERROR reading PATCH file");
  }
  numPatches=len;
  patches = new Patch[len];
  //cerr<<numPatches<<endl;

  rewind(fd);

  int i=0;
  while (!feof(fd)) {
    fgets(tmp, 1000, fd);        // read the line
    strcpy(keyword, "\0");       // Reset ketword

    int n=sscanf(tmp, "%s %s %s %s", keyword, pres, res1, res2);

    if (n>=3 && (!strcmp(keyword, "patch")|| !strcmp(keyword, "PATCH"))) {

      strcpy(patches[i].pres, pres);
      delim = strchr(res1, ':');
      strncpy(patches[i].segname1, res1, delim-res1);
      patches[i].segname1[delim-res1]='\0'; // terminate string
      resid=strchr(res1, ':');
      //      cerr<<"segname="<<patches[i].segname1<<" resid="<<resid<<endl;
      resid++;
      patches[i].resid1 = atoi(resid);
      // if we have a patch involving more than one residue...
      if (n==4) {
        delim = strchr(res2, ':');
        strncpy(patches[i].segname2, res2, delim-res2);
        patches[i].segname2[delim-res2]='\0';
        resid = strchr(res2, ':');
        resid++; // skip the ':'
        patches[i].resid2 = atoi(resid);
      }
      else {
        patches[i].resid2 = -1;
        strcpy(patches[i].segname2, "NONE");
      }
      if (ferror(fd)) error_exit("ERROR reading atom selection file");
      i++;
    }
  }
  fclose(fd);
}

//  Lookup atom id from segment, residue, and index in residue
int* HBonds::get_atoms_from_segid_and_residue(const char *segid, int resid, int *natoms) const 
{
  int *tmp = new int[numAtoms];
  *natoms=0;
  for (int i=0; i<numAtoms; i++) {
    if (!strcmp(atomNames[i].segname, segid) && atomNames[i].resid==resid) {
      tmp[(*natoms)++] = i;
    }
  } 
  int *index = new int[*natoms];
  for(int j=0; j<(*natoms); j++) {
    index[j] = tmp[j];
  }
  delete [] tmp;
  return index;
}
