#include "auxil.h"
#include <sys/time.h>
#include <sys/stat.h>
#include <string.h>
#include <math.h>
#include <iostream.h>
#include <stdlib.h>
#include <ctype.h>
#include "Vector.h"
#include "ComputeNonbonded.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

// get the time of day from the system clock, and store it (in seconds)
//static double time_of_day(void) {
//  struct timeval tm;
//  struct timezone tz;

//  gettimeofday(&tm, &tz);
//  return((double)(tm.tv_sec) + (double)(tm.tv_usec)/1000000.0);
//}

void usage_exit(const char *s, const char *argv0) {
  fprintf(stderr, "%s\n",s); 
  fprintf(stderr, "Usage: %s ", argv0);
  fprintf(stderr, "<-pdb file|-dcd file> -psf file -par file [-avg n>] \n");
  fprintf(stderr, "[-beg firstframe] [-end [lastframe|'last']] \n");
  fprintf(stderr, "<-bond|-angl|-dihe|-impr|-vdw|-elec|-nonb|-kin|-conf|-all|-hbon>,\n");
  exit(0);
}

void bad_energy_select_exit() {
  fprintf(stderr, "ERROR: Must specify energy or force smooth coordinates to be computed.\n");
  fprintf(stderr, "Allowed energy/force selections are:\n");
  fprintf(stderr, "{-bond -angl -dihe -impr -vdw -elec -nonb -kin -conf -tot -all,\n");
  fprintf(stderr, "-fconf -fvdw -felec -ftot}\n");
  fprintf(stderr, "Trajectory smoothing is done with -smooth\n");
  exit(0);
}

void error_exit(const char *s) {
  fprintf(stderr, "%s\n",s);
  exit(0);
}

void NAMD_die(const char *s) {
  fprintf(stderr, "%s\n",s);
  exit(0);
}

// Trims trailing whitespaces
void eatwhiteleft(char *s) {
  int i,j=0;
  char *buf = new char[strlen(s)];

  for(i=0; i<(int)strlen(s); i++) {
    if ((j>0) || !isspace(s[i]))
      {
	buf[j] = s[i];
	j++;
      }
  }
  buf[j]='\0';
  strcpy(s, buf);
  delete [] buf;
}

// Trims trailing whitespaces
void trimfirst(char* s) {
  int i;
  int len = strlen(s);
  char* buf = new char[len];
  for (i=0; i<len-1; i++) {
    buf[i] = s[i+1];
  }
  buf[len-1]='\0';
  strncpy(s, buf, len);

  delete [] buf;
}

FILE * open_write(const char *E_file_name) {
  return fopen(E_file_name, "w+b");
}

void close_write(FILE* fd) {
  fclose(fd);
}
void write_E_file_header(FILE* fd, int natoms, int *atomsel, char* seltext,
			 char* selected_energy, int navg, int self)
{
  fprintf(fd, "type:          %s \n", selected_energy);
  fprintf(fd, "natoms:        %d \n", natoms);
  fprintf(fd, "avg win:       %d \n", navg);
  fprintf(fd, "self only:     %d \n", self);
  fprintf(fd, "selectiontext: %s \n", seltext);
  fprintf(fd, "indexes:   ");
  for (int i=0; i<natoms; i++)
    fprintf(fd, "%d ", atomsel[i]);
  fprintf(fd, "\n");
}


void write_F_file_header(FILE* fd, int natoms, int *atomsel, char* seltext,
			 char* selected_energy, int navg, int self)
{
  fprintf(fd, "type:          %s \n", selected_energy);
  fprintf(fd, "natoms:        %d \n", natoms);
  fprintf(fd, "avg win:       %d \n", navg);
  fprintf(fd, "self only:     %d \n", self);
  fprintf(fd, "selectiontext: %s \n", seltext);
  fprintf(fd, "indexes:   ");
  for (int i=0; i<natoms; i++)
    fprintf(fd, "%d ", atomsel[i]);
  fprintf(fd, "\n");
}

void write_HB_file_header(FILE* fd, int natoms, int *atomsel, char* seltext,
			  char* selected_energy, int self)
{
  fprintf(fd, "type:          %s \n", selected_energy);
  fprintf(fd, "natoms:        %d \n", natoms);
  fprintf(fd, "self only:     %d \n", self);
  fprintf(fd, "selectiontext: %s \n", seltext);
  fprintf(fd, "indexes:   ");
  for (int i=0; i<natoms; i++)
    fprintf(fd, "%d ", atomsel[i]);
  fprintf(fd, "\n");
}

void write_HB_step(FILE* fd, int numHbonds, int Frame, hbond *hb)
{
  int i=0;
  fprintf(fd, "frame:     %d \n", Frame);
  fprintf(fd, "numHBonds: %d \n", numHbonds);

  struct hbond *ptr;
  ptr = hb;
  while (ptr!=NULL) {
    fprintf(fd, "{%f %f %f} {%f %f %f} %f  ", ptr->posDon->x, ptr->posDon->y, ptr->posDon->z, 
	    ptr->posAcc->x, ptr->posAcc->y, ptr->posAcc->z, ptr->E);
    ptr = ptr->next;
    i++;
  }
  fprintf(fd, "\n");
   
  if (i!=numHbonds) error_exit("ERROR: Wrong number of Hbonds in hbondlist!");
  if (ferror(fd)) error_exit("ERROR writing F_file");
}

/*************************************************************/
/***                                                       ***/
/***   This is the Callback function for the               ***/
/***   SlidingWindowAverage class, which is called         ***/
/***   data from the ringbuffer shall be processed.        ***/
/***                                                       ***/
/*************************************************************/

int write_avg_energies(void* param, const void* data, int Frame)
{
  E_clientdata* p = (E_clientdata*) param;
  float* avg = (float*) data;
  int i, min_i=0, max_i=0;
  float min=avg[0];
  float max=avg[0];

  for (int j=0; j<p->Natoms; j++) {
    // logarithmic scale:
    if (p->logscale) avg[j] = log(avg[j]);

    // Determine min and max:
    if (avg[j]>max) {
      max = avg[j];
      max_i = j;
    }
    if (avg[j]<min) {
      min = avg[j];
      min_i = j;
    }
  }

  fprintf(p->fd, "frame:     %d \n", Frame);
  fprintf(p->fd, "min/max i: %d %d\n", min_i, max_i);
  fprintf(p->fd, "min/max  : %f %f\n", avg[min_i], avg[max_i]);
  for (i=0; i<p->Natoms; i++)
    fprintf(p->fd, "%f ", avg[i]);

  fprintf(p->fd, "\n");

  if (ferror(p->fd)) error_exit("ERROR writing E_file");
  return 1;
}
                                                                                              
int write_avg_coords(void* param, const void* data, int Frame)
{
  Coor_clientdata* p = (Coor_clientdata*) param;
  float *avgcoor = (float*) data;
  float *X = new float[p->Natoms];
  float *Y = new float[p->Natoms];
  float *Z = new float[p->Natoms];
  int i, j;
                                                                                              
  for(i=0, j=0; i < p->Natoms; i++) {
                                                                                              
      X[i]=avgcoor[j++];
      Y[i]=avgcoor[j++];
      Z[i]=avgcoor[j++];
      //cout << j <<":  "<<avgcoor[j] << "  " << X[i] << "  " << p->Natoms<< endl;
  };
                                                                                              
  p->dcd->write(p->fd, Frame, X, Y, Z);
  delete [] X;
  delete [] Y;
  delete [] Z;
                                                                                              
  return 1;
}
                                                                                              
int write_avg_forces(void* param, const void* data, int Frame)
{
  F_clientdata* p = (F_clientdata*) param;
  Vector* avg = (Vector*) data;
  int i, min_i=0, max_i=0;
  double min=avg[0].length();
  double max=min;
  double len;

  for (int j=0; j<p->Natoms; j++) {
    len = avg[j].length();
    // Determine min and max:
    if (len>max) {
      max = len;
      max_i = j;
    }
    if (len<min) {
      min = len;
      min_i = j;
    }
  }

  fprintf(p->fd, "frame:     %d \n", Frame);
  fprintf(p->fd, "min/max i: %d %d\n", min_i, max_i);
  fprintf(p->fd, "min/max:   %f %f\n", min, max);
  fprintf(p->fd, "total:     %f %f %f \n", p->total->x, p->total->y, p->total->z);
  for (i=0; i<p->Natoms; i++)
    fprintf(p->fd, "%f %f %f  ", avg[i].x, avg[i].y, avg[i].z);

  fprintf(p->fd, "\n");

  if (ferror(p->fd)) error_exit("ERROR writing F_file");
  return 1;
}
     

FILE* open_read(const char *filename) {
  return fopen(filename, "r+b");
}

void close_read(FILE* fd) {
  fclose(fd);
}


/***************************************************************************
 Fopen(char *Filename, char *mode):  similar to fopen(filename,mode) except
 it checks for compressed file names too.
 For example:  Fopen("config");
   This will first look for the filename "config" (and return "r" file handle
   if it is found).
   Then it will look for "config.Z" (and run "zcat config.Z", returning
   a file handle to the uncompressed data if found).
   Then it will look for "config.gz" (and run "gzip -d -c config.gz", returning
   a file handle to the uncompressed data if found).
 ***************************************************************************/
FILE *Fopen	(const char *filename, const char *mode)
{
  struct stat buf;
  // check if basic filename exists (and not a directory)

#if defined(NOCOMPRESSED)
  if (!stat(filename,&buf))
    {
      return(fopen(filename,mode));
    }
#else
  if (!stat(filename,&buf))
    {
      if (!S_ISDIR(buf.st_mode))
	return(fopen(filename,mode));
    }
  // check for a compressed file
  char *realfilename;
  char *command;
  FILE *fout;
  command = (char *)malloc(strlen(filename)+25);
  // check for .Z (unix compress)
  sprintf(command,"zcat %s.Z",filename);
  realfilename = command+5;
  cout << "Command = " << command << "\n" << endl;
  cout << "Filename.Z = " << realfilename << "\n" << endl;
  if (!stat(realfilename,&buf))
	{
	if (!S_ISDIR(buf.st_mode))
		{
		fout = popen(command,mode);
		// on HP-UX, the first character(s) out of pipe may be
		// garbage!  (Argh!)
		int C;
		do
		  {
		  C = fgetc(fout);
		  // iout << "C is " << C << "\n" << endi;
		  if (isalnum(C) || isspace(C))
			{
			ungetc(C,fout);
			C = -1;	// outta loop
			}
		  } while(C != -1);
		free(command);
		return(fout);
		}
	}
  // check for .gz (gzip)
  sprintf(command,"gzip -d -c %s.gz",filename);
  realfilename = command+11;
  cout << "Command = " << command << "\n" << endl;
  cout << "Filename.gz = " << realfilename << "\n" << endl;
  if (!stat(realfilename,&buf))
	{
	if (!S_ISDIR(buf.st_mode))
		{
		fout = popen(command,mode);
		// on HP-UX, the first character(s) out of pipe may be
		// garbage!  (Argh!)
		int C;
		do
		  {
		  C = fgetc(fout);
		  // iout << "C is " << C << "\n" << endi;
		  if (isalnum(C) || isspace(C))
			{
			ungetc(C,fout);
			C = -1;	// outta loop
			}
		  } while(C != -1);
		free(command);
		return(fout);
		}
	}
  free(command);
#endif /* !defined(NOCOMPRESSED) */

  return(NULL);
} /* Fopen() */

/***************************************************************************
 Fclose(FILE *fout):  similar to fclose(fout) except it first checks if the
 file handle fout is a named pipe.
 ***************************************************************************/
int	Fclose	(FILE *fout)
{
  int rc = -1;
#if !defined(NOCOMPRESSED)
  rc = pclose(fout);
#endif
  if (rc == -1)	// stream not associated with a popen()
    {
    rc = fclose(fout);
    }
  return rc;
} /* Fclose() */

