#ifndef SLIDEWIN_H
#define SLIDEWIN_H

// Sliding window average over frames: width = 1+2*navg
// (center element + navg values to the left and to the right)
// to conserve the number of frames the first and last navg values
// are filled with the navg-th value.

// For the data type T there must be defined:
// operator+=
// operator/=
// operator>
// operator<

#include "Ringbuf.h"

// Define the pointer to the callback function that is called when the 
// data in the ringbuffer shall be processed.
typedef int (*Slidewin_callback_function)(void *clientdata, const void *ringdata, int curr_frame);

template<typename T> 
class SlidingWindowAverage {
 private:
  int half_width;
  int array_size;
  int num_frames;
  int fill_level;
  int padding;
  int cropping;
  int finished;
  Ringbuf ringbuf;
  T* avg;  // The averages are held in here
  T* tmp;

  // when it's time to process data, call:
  // (*callback_function)(clientdata, ringdata, curr_frame);
  
  Slidewin_callback_function callback_function;

 public:
  void *clientdata;
  SlidingWindowAverage(int navg, int array_size);
  ~SlidingWindowAverage();

  void add_data(int frame, T *rawdata);
  void finish();
  //  void set_logarithmic_scale() { logarithmic=1; };
  void set_padding() { padding=1; };
  void set_ramping() { padding=0; cropping=0; };
  void set_cropping() { cropping=1; };
  void set_callback(void *, Slidewin_callback_function);
  void average(float* tmp);
  void average(double* tmp);
  void average(Vector* tmp);
  void scale(float* dummy);
  void scale(double* dummy);
  void scale(Vector* dummy);
};


// sliding window average over frames: width = 1+2*navg
// (center element + navg values to the left and to the right)
// to conserve the number of frames the first and last navg values
// are filled with the navg-th value.
                                                                                              
template <typename T>
SlidingWindowAverage<T>::SlidingWindowAverage(int navg, int array_size) {
  this->array_size=array_size;
  half_width=navg;
  fill_level=0;
  finished=0;
  size_t elem_size=sizeof(T);
  avg = new T[array_size];    
  tmp = new T[array_size];    

  // Initialize the ringbuffer with the windowsize and the size to store one of the arrays
  ringbuf.init(2*navg+1, array_size*elem_size);

  padding=0; // ramping by default
  cropping=0;
}

template <typename T>
SlidingWindowAverage<T>::~SlidingWindowAverage() {
  delete [] avg;
  avg = NULL;
  delete [] tmp;
  tmp = NULL;
}

// Sets the pointer to the callback function
template <typename T>
void SlidingWindowAverage<T>::set_callback(void *cl_data, Slidewin_callback_function callback_func) {
  clientdata=cl_data;
  callback_function=callback_func;
}

// Takes the current frame number and a pointer to the data that should be 
// added to the ringbuffer.
// 
template <typename T>
void SlidingWindowAverage<T>::add_data(int curr_frame, T *rawdata) {

  this->num_frames = curr_frame;
  // Initialize the array to hold the averages
  memset((void *)avg, 0, array_size*sizeof(T));

  // fill in next element in ringbuf
  ringbuf.set_next( rawdata );
  fill_level++;

  // when buffer full get window avg
  if (curr_frame >= 0)
     {
      for (int a=0; a<2*half_width+1; a++) {
	ringbuf.get_next(tmp);

	// Now take the average
	average(tmp);
      }
      scale(avg);
    }

  //ramping
  if (!padding && !cropping && half_width>0) { 

      if (curr_frame >= half_width) {           // curr_frame=6
	  // Data processing by the callback function:
	  callback_function(clientdata, avg, curr_frame-half_width);
	  if (fill_level>=2*half_width+1) { 
	      fill_level--;
	      //cout<<curr_frame-half_width<<"-\n";
	  } else {
	      // Ramp up
	      //cout<<curr_frame-half_width<<">\n";
	  }
      }
  }

  if (padding && half_width>0) {
      // write the first navg B-values (eg. nframes=100, half_width=3)
      if (curr_frame == 2*half_width) {           // curr_frame=6
	  for (int c=0; c<half_width; c++) {      // 4 times
	      callback_function(clientdata, avg, c); // pos 4
	      //cout<<c<<"+\n";
	  }
	  fill_level--;
      }
      
      // write once for intermediate steps
      if (curr_frame >= 2*half_width) {  // curr_frame>6 && curr_frame<100
	  callback_function(clientdata, avg, curr_frame-half_width);
	  //cout<<curr_frame-half_width<<"*\n";
	  fill_level--;
      }
      
      // write the last navg B-values
  }

  if (cropping || half_width==0) {
      if (curr_frame >= 2*half_width) {  // curr_frame>6 && curr_frame<100
	  callback_function(clientdata, avg, curr_frame-2*half_width);
	  fill_level--;
	  //cout<<curr_frame-2*half_width<<"*\n";
      }
  }
  
}

template <typename T>
void SlidingWindowAverage<T>::finish() {

  //ramping
  if (!padding && !cropping && half_width>0) { 
      // take remaining averages (ramp down)
      for (int c=0; c<half_width; c++) {  
	  memset((void *)avg, 0, array_size*sizeof(T));
	  ringbuf.shift_to_origin(); // shift to origin
	  ringbuf.shift(c+1);
	  
	  for (int a=0; a<fill_level; a++) {
	      ringbuf.get_next(tmp);
	      
	      // Now take the average
	      average(tmp);
	  }
	  scale(avg);

	  // Data processing by the callback function:
	  callback_function(clientdata, avg, num_frames-half_width+c+1); // pos 96
	  fill_level--;
	  //cout<<num_frames-half_width+c+1<<"<\n";
      }
  }
  
  if (padding && half_width>0) {
      // write the last navg B-values
      for (int c=0; c<half_width; c++) {     // 3 times
	  callback_function(clientdata, avg, num_frames-half_width+c+1); // pos 96
	  //cout<<num_frames-half_width+c+1<<"-\n";
      }
  }
}

// Compute the average
template <typename T>
void SlidingWindowAverage<T>::average(float* tmp) {
  for (int i=0; i<array_size; i++)
    avg[i] += tmp[i];
}

template <typename T>
void SlidingWindowAverage<T>::average(double* tmp) {
  for (int i=0; i<array_size; i++)
    avg[i] += tmp[i];
}

template <typename T>
void SlidingWindowAverage<T>::average(Vector* tmp) {
  for (int i=0; i<array_size; i++) {
    avg[i].x += tmp[i].x;
    avg[i].y += tmp[i].y;
    avg[i].z += tmp[i].z;
  }
}


template <typename T>
void SlidingWindowAverage<T>::scale(float* dummy) {
  for (int j=0; j<array_size; j++)
    avg[j] /= ((T) fill_level);
}

template <typename T>
void SlidingWindowAverage<T>::scale(double* dummy) {
  for (int j=0; j<array_size; j++)
    avg[j] /= ((T) fill_level);
}

template <typename T>
void SlidingWindowAverage<T>::scale(Vector* dummy) {
  for (int j=0; j<array_size; j++)
    avg[j] /= ((double) fill_level);
}


#endif
