#include <iostream.h>
#include <math.h>
#include "utilities.h"
#include "Parameters.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

//  struct hbond_pair_params is used to form a linked list of the
//  hbond parameters for a pair of atoms

struct hbond_pair_params
{
  char donname[11];
  char accname[11];
  double E;
  double R;
  struct hbond_pair_params *next;
};

struct donacc_type
{
  char type[11];
  char pos[100];
  Index index;
  struct donacc_type *left;
  struct donacc_type *right;
};


/************************************************************************/
/*                  */
/*      FUNCTION add_hbond_pair_param      */
/*                  */
/*   INPUTS:                */
/*  buf - line containing the vdw_pair information      */
/*                  */
/*  this function adds a hbond_pair parameter to the current          */
/*   parameters.              */
/*                  */
/************************************************************************/

void Parameters::add_hbond_pair_param(char *buf)

{
  char donname[11], accname[11]; //  atom type of paramter
  double Emin;       //  Emin value for this hbond
  double Rmin;       //  R(Emin) value for this hbond
  int read_count=0;  //  count returned by sscanf
  struct hbond_pair_params *new_node;  //  new node for list
  struct donacc_type *new_donor;  //  new node for list
  struct donacc_type *new_accep;  //  new node for list

  //****** BEGIN CHARMM/XPLOR type changes
  /*  Parse up the line with sscanf        */
  if (paramType == paraXplor)
  {
    /* read XPLOR format */
    read_count=sscanf(buf, "%*s %s %s %lf %lf\n", donname, accname, 
       &Emin, &Rmin);
  }
  else if (paramType == paraCharmm)
  {
    /* read CHARMM format */
    read_count=sscanf(buf, "%s %s %lf %lf\n", donname, accname, 
       &Emin, &Rmin);
  }

  cout << donname << "  " << accname << "  " << Emin << "  " << Rmin << endl;

  /*  Check to make sure we got what we expected      */
  if ((read_count != 4) && (paramType == paraXplor))
  {
    char err_msg[512];

    sprintf(err_msg, "BAD HBOND FORMAT IN XPLOR PARAMETER FILE\nLINE=*%s*\n", buf);
    error_exit(err_msg);
  }
  else if ( ((read_count != 4)) && (paramType == paraCharmm))
  {
    char err_msg[512];

    sprintf(err_msg, "BAD HBOND FORMAT IN CHARMM PARAMETER FILE\nLINE=*%s*\n", buf);
    error_exit(err_msg);
  }

  /*  Allocate a new node            */
  new_node = new hbond_pair_params;
  new_donor = new donacc_type;
  new_accep = new donacc_type;

  if (new_node == NULL || new_donor == NULL || new_accep == NULL)
  {
    error_exit("memory allocation failed in Parameters::add_hbond_param");
  }

  /*  Assign the values to the new node        */
  strcpy(new_node->donname, donname);
  strcpy(new_node->accname, accname);

  // convert and assign the values:
  double r2 = Rmin*Rmin;
  double r6  = pow(r2, 3);
  //double r12 = r6*r6;
  //new_node->E = -Emin*r12;
  //new_node->R = -2*Emin*r6;
  new_node->E = -0.5*Emin*r6;
  new_node->R = -0.75*Emin*r2;
  new_node->next = NULL;

  /*  Add the new node into the tree        */
  add_to_hbond_pair_list(new_node);


  strcpy(new_donor->type, donname);
  strcpy(new_accep->type, accname);
  new_donor->left=NULL;
  new_donor->right=NULL;
  new_accep->left=NULL;
  new_accep->right=NULL;

  donor_type_tree=add_to_donacc_type_tree(new_donor, donor_type_tree);
  accep_type_tree=add_to_donacc_type_tree(new_accep, accep_type_tree);

  return;
}
/*      END OF FUNCTION add_hbond_pair_param      */
 

/************************************************************************/
/*                                                                      */
/*      FUNCTION add_to_hbond_pair_list                                 */
/*                                                                      */
/*   INPUTS:                                                            */
/*  new_node - node to be added to list                                 */
/*                                                                      */
/*  This function adds a link to the end of the hbond_pair_list list    */
/*                                                                      */
/************************************************************************/

void Parameters::add_to_hbond_pair_list(struct hbond_pair_params *new_node)
{
  static struct hbond_pair_params *tail=NULL;
  struct hbond_pair_params *ptr;
  int compare_code;
  

  //  If the list was empty, then just make the new node the list
  if (hbond_pairp == NULL)
  {
     hbond_pairp = new_node;
     tail = new_node;
     return;
  }
  
  ptr = hbond_pairp;

  //  Now check the list to see if we have a duplicate entry
  while (ptr!=NULL)
  {
      /*  Compare atom 1            */
      compare_code = strcasecmp(new_node->donname, ptr->donname);
      
      if (compare_code == 0)
      {
	/*  Atom 1 is the same, compare atom 2      */
	compare_code = strcasecmp(new_node->accname, ptr->accname);
	
	if (compare_code==0)
	{
	  /*  Found a duplicate.  Print out a warning   */
	  /*  message, assign the values to the current   */
	  /*  node in the tree, and then free the new_node*/
	  if ((ptr->E != new_node->E) || (ptr->R != new_node->R))
	  {
	    cout  << "DUPLICATE hbond PAIR ENTRY FOR "
		  << new_node->donname << "-"
		  << new_node->accname
		  << "\nPREVIOUS VALUES  E=" << ptr->E
		  << " R=" << ptr->R
		  << "\n   USING VALUES  E=" << new_node->E
		  << " R=" << new_node->R
		  << "\n" << endl;

	    ptr->E=new_node->E;
	    ptr->R=new_node->R;

	  }
	  delete new_node;
	  return;
	}
      }

      ptr = ptr->next;
  }

  //  We didn't find a duplicate, so add this node to the end
  //  of the list
  tail->next = new_node;
  tail = new_node;
}
/*      END OF FUNCTION add_to_hbond_pair_list    */


/************************************************************************/
/*                  */
/*      FUNCTION add_to_donacc_type_tree    */
/*                  */
/*   INPUTS:                */
/*  new_node - new node to be added to the tree      */
/*  tree - tree to add the node to          */
/*                  */
/*  This is a recursive function that adds a node to the    */
/*   binary search tree of donacc parameters        */
/*                  */
/************************************************************************/

struct donacc_type* Parameters::add_to_donacc_type_tree(struct donacc_type *new_node,
                 struct donacc_type *tree)
   
{
  int compare_code;  //  Return code from strcasecmp

  if (tree == NULL)
      return(new_node);
   
  compare_code = strcasecmp(tree->type, new_node->type);

  /*  Check to see if we have a duplicate        */
  if (compare_code==0)
  {
    /*  We have a duplicate.  So print out a warning   */
    /*  message and then free the new_node    */

      // cout  << "DUPLICATE don/acc ENTRY FOR " << tree->type
      //  << "\n" << endl;

    delete new_node;

    return(tree);
  }

  /*  Otherwise, if the new node is less than the head of    */
  /*  the tree, add it to the left child, and if it is greater  */
  /*  add it to the right child          */
  if (compare_code < 0)
  {
    tree->left = add_to_donacc_type_tree(new_node, tree->left);
  }
  else
  {
    tree->right = add_to_donacc_type_tree(new_node, tree->right);
  }
  return(tree);
}

/*      END OF FUNCTION add_to_donacc_type_tree  */

/************************************************************************/
/*                  */
/*      FUNCTION index_donacc        */
/*                  */
/*   INPUTS:                */
/*  tree - The tree that is to be indexed        */
/*  index - index to start with          */
/*                  */
/*  This is a recursive routine that will traverse the binary tree  */
/*   of donacc_type names, assigning an index to each one, and copying     */
/*   the data from the binary tree to the array that will be used from  */
/*   here on.                */
/*                  */
/************************************************************************/

Index Parameters::index_donacc(struct donacc_type *tree, Index index)

{
  //  If the tree is empty, do nothing
  if (tree==NULL)
    return(index);

  //  If I have a left subtree, populate it first
  if (tree->left != NULL)
  {
    index=index_donacc(tree->left, index);
  }

  //  Assign the index and copy the data to the array
  tree->index = index;

  cout  << "Parameters: Stored don/acc type " << index << ": ";
      cout  << tree->type << endl;

  index++;

  //  If I have a right subtree, index it
  if (tree->right != NULL)
  {
    index=index_donacc(tree->right, index);
  }

  return(index);
}
/*      END OF FUNCTION index_donacc      */


/************************************************************************/
/*                                                                      */
/*      FUNCTION get_donacc_index                                       */
/*                                                                      */
/*   INPUTS:                                                            */
/*  tree - The tree that is to be searched                              */
/*  type - string containing the type name to be compared               */
/*                                                                      */
/*  This routine will traverse the binary tree of indexed               */
/*   donacc_type names, comparing the names with type.                  */
/*   It returns the index of the matched type or -1                     */
/*   if no match was found.                                             */
/*                                                                      */
/************************************************************************/

int Parameters::get_donacc_index(struct donacc_type *tree, const char *type)

{
  donacc_type *ptr;    //  Current location in tree
  int found=0;         //  Flag 1-> found a match
  int comp_code;       //  return code from strcasecmp  
  ptr=tree;

  /*  While we haven't found a match and we're not at the end  */
  /*  of the tree, compare the donacc type passed in with the tree  */
  while (!found && (ptr!=NULL))
  {
    comp_code = donacctypecmp(ptr->type, type);

    if ( !comp_code )
    {
       found = 1;
    }
    else if (comp_code < 0) 
    {
      /*  Go left          */
      ptr=ptr->left;
    }
    else
    {
      /*  Go right          */
      ptr=ptr->right;
    }
  }

  /*  If we found a match, return the index      */
  if (found)
  {
    return(ptr->index);
  }
  else
  {
    return(-1);
  }

}
/*      END OF FUNCTION get_donacc_index      */

/************************************************************************/
/*                                                                      */
/*      FUNCTION donacctypecmp                                          */
/*                                                                      */
/*   INPUTS:                                                            */
/*  type1 - pointer to the first type                                   */
/*  type2 - pointer to the second type                                  */
/*                                                                      */
/*                                                                      */
/*   OUTPUTS:                                                           */
/*  returns 1 if type match, 0 else.                                    */
/*                                                                      */
/*  This helper function compares two strings containing                */
/*  donor/acceptor type names. The wildcard '%' for any character       */
/*  is recognized.                                                      */
/*                                                                      */
/************************************************************************/

int donacctypecmp(const char* type1, const char* type2)
{
    char *ptr1, *ptr2;
    int nomatch=1;
    char* u1 = stringtoupper(stringdup(type1));
    char* u2 = stringtoupper(stringdup(type2));
    ptr1 = u1;
    ptr2 = u2;
    int i=0;
    while (!(*ptr1=='\0' && *ptr2=='\0')) {
	i++;
	if ((*ptr1 == '%' && *ptr2!='\0') || (*ptr1!='\0' && *ptr2 == '%')) {
	    nomatch = 0;
	    //   cerr << i <<" wild:    " << *ptr1 << "  " << *ptr2 << endl;
	    ptr1++; ptr2++;
	    continue;
	}
	if (*ptr1 != *ptr2) { 
	    nomatch=1; 
	    //cerr << i << " nomatch:  " << *ptr1 << "  " << *ptr2 << endl;
	    break;
	}
	nomatch = 0;
	//cerr << i << "  " << *ptr1 << "  " << *ptr2 << endl;
	ptr1++; ptr2++;
    } 
    if (nomatch) { nomatch=strcasecmp(u1, u2); }

    delete [] u1;
    delete [] u2;

    return nomatch;
}
/*      END OF FUNCTION donacctypecmp      */


/************************************************************************/
/*                                                                      */
/*      FUNCTION get_hbond_pair_params                                  */
/*                                                                      */
/*   INPUTS:                                                            */
/*  donindex - donor type index for atom 1                                        */
/*  accindex - acceptor type index for atom 2                                        */
/*  E - E value to populate                                             */
/*  R - R value to populate                                             */
/*                                                                      */
/*   OUTPUTS:                                                           */
/*  If a match is found, A, B, are populated and a                      */
/*   1 is returned.  Otherwise, a 0 is returned.                        */
/*                                                                      */
/*  This function finds a set of hbond_pair paramters.  It is given     */
/*   the two types of atoms involved.  This is the only paramter for    */
/*   which a match is NOT guaranteed.  There will only be a match if    */
/*   there are specific hbond parameters for the two atom types */
/*   involved.                                                          */
/*                                                                      */
/************************************************************************/

int Parameters::get_hbond_pair_params(int donindex, int accindex, double *E, 
				      double *R) const
{
  IndexedHbondPair *ptr;   //  Current location in tree
  int found=0;             //  Flag 1-> found a match

  ptr=hbond_pair_tree;

  /*  While we haven't found a match and we're not at the end  */
  /*  of the tree, compare the bond passed in with the tree  */
  while (!found && (ptr!=NULL))
  {
    if ( (donindex == ptr->donindex) && (accindex == ptr->accindex) )
    {
       found = 1;
    }
    else if ( (donindex < ptr->donindex) || 
        ( (donindex==ptr->donindex) && (accindex < ptr->accindex) ) )
    {
      /*  Go left          */
      ptr=ptr->left;
    }
    else
    {
      /*  Go right          */
      ptr=ptr->right;
    }
  }

  /*  If we found a match, assign the values      */
  if (found)
  {
    *E = ptr->E;
    *R = ptr->R;
    //cerr << "donor match  " << dtype << "  " << dtype2 << endl;
    //cerr << "acceptor match  " << atype << "  " << atype2 << endl;
    return(1);
  }
  else
  {
    return(0);
  }

}
/*      END OF FUNCTION get_hbond_pair_params    */


/************************************************************************/
/*                                                                      */
/*      FUNCTION free_hbond_pair_list                                   */
/*                                                                      */
/*  This function frees the hbond_pair_list                             */
/*                                                                      */
/************************************************************************/

void Parameters::free_hbond_pair_list()
{
   struct hbond_pair_params *ptr, *next;
   
   ptr=hbond_pairp;
   
   while (ptr != NULL)
   {
      next = ptr->next;
      
      delete ptr;
      
      ptr = next;
   }
   
   hbond_pairp = NULL;
}
/*      END OF FUNCTION free_hbond_pair_list    */


/************************************************************************/
/*                  */
/*      FUNCTION free_donacc_tree        */
/*                  */
/*   INPUTS:                */
/*  type_ptr - pointer to type_tree tree to free        */
/*                  */
/*  this is a recursive function that is used to free the memory    */
/*   allocated for a don/acc type tree.  It makes recursive calls to    */
/*   free the left an right subtress, and then frees the head.  It is   */
/*   only called by the destructor          */
/*                  */
/************************************************************************/

void Parameters::free_donacc_tree(struct donacc_type *type_ptr)

{
  if (type_ptr->left != NULL)
  {
    free_donacc_tree(type_ptr->left);
  }

  if (type_ptr->right != NULL)
  {
    free_donacc_tree(type_ptr->right);
  }

  delete type_ptr;

  return;
}
/*      END OF FUNCTION free_donacc_tree      */


/************************************************************************/
/*                  */
/*      FUNCTION free_hbond_pair_tree        */
/*                  */
/*   INPUTS:                */
/*  hbond_pair_ptr - pointer to type_tree tree to free        */
/*                  */
/*  this is a recursive function that is used to free the memory    */
/*   allocated for a hbond_pair parameter tree.  It makes recursive calls to    */
/*   free the left an right subtress, and then frees the head.  It is   */
/*   only called by the destructor          */
/*                  */
/************************************************************************/

void Parameters::free_hbond_pair_tree(IndexedHbondPair *hbond_pair_ptr)

{
  if (hbond_pair_ptr->left != NULL)
  {
    free_hbond_pair_tree(hbond_pair_ptr->left);
  }

  if (hbond_pair_ptr->right != NULL)
  {
    free_hbond_pair_tree(hbond_pair_ptr->right);
  }

  delete hbond_pair_ptr;

  return;
}
/*      END OF FUNCTION free_hbond_pair_tree      */


/************************************************************************/
/*                  */
/*      FUNCTION convert_hbond_pairs      */
/*                  */
/*  This function converts the linked list of hbond_pairs indexed by    */
/*  atom name into a binary search tree of parameters stored by hbond   */
/*  type index.  This tree is what will be used for real when searching */
/*  for parameters during the simulation.        */
/*                  */
/************************************************************************/

void Parameters::convert_hbond_pairs()
   
{
   int donindex, accindex;  //  Indexes for the two atoms
   IndexedHbondPair *new_node;  //  New node for tree
   struct hbond_pair_params *ptr, *next;  //  Pointers for traversing list
   
   ptr = hbond_pairp;
   
   //  Go down then entire list and insert each node into the 
   //  binary search tree
   while (ptr != NULL)
   {
      new_node = (IndexedHbondPair *) malloc(sizeof(IndexedHbondPair));

      if (new_node == NULL)
      {
	error_exit("memory allocation failed in Parameters::convert_hbond_pairs");
      }
      
      //  Get the donor and acceptor type indexes for the two atoms.
      donindex = get_donacc_index(donor_type_tree, ptr->donname);
      accindex = get_donacc_index(accep_type_tree, ptr->accname);
      
      if ((donindex<0) || (accindex<0))
      {
	 char err_msg[257];

         sprintf(err_msg, "No H-Bond Parameters found for donor %i:%s and acceptor %i:%s", donindex, ptr->donname, accindex, ptr->accname);
         error_exit(err_msg);	  
      }
           
      new_node->donindex = donindex;
      new_node->accindex = accindex;
      new_node->E = ptr->E;
      new_node->R = ptr->R;
      
      new_node->left = NULL;
      new_node->right = NULL;
      
      //  Add it to the tree
      hbond_pair_tree = add_to_indexed_hbond_pairs(new_node, hbond_pair_tree);
      
      //  Free the current node and move to the next
      next = ptr->next;
      
      delete ptr;
      
      ptr = next;
   }
   
   hbond_pairp = NULL;
}
/*      END OF FUNCTION convert_hbond_pairs    */

/************************************************************************/
/*                  */
/*      FUNCTION add_to_indexed_hbond_pairs    */
/*                  */
/*   INPUTS:                */
/*  new_node - new node to be added to the tree      */
/*  tree - tree to add the node to          */
/*                  */
/*  This is a recursive function that adds a node to the    */
/*   binary search tree of hbond_pair parameters        */
/*                  */
/************************************************************************/

IndexedHbondPair* Parameters::add_to_indexed_hbond_pairs(IndexedHbondPair *new_node,
                 IndexedHbondPair *tree)
   
{
   if (tree == NULL)
      return(new_node);

   if ( (new_node->donindex < tree->donindex) || 
        ((new_node->donindex == tree->donindex) 
	 && (new_node->accindex < tree->accindex)) )
   {
      tree->left = add_to_indexed_hbond_pairs(new_node, tree->left);
   }
   else
   {
      tree->right = add_to_indexed_hbond_pairs(new_node, tree->right);
   }
   
   return(tree);
}
/*      END OF FUNCTION add_to_indexed_hbond_pairs  */

