/**
***  Copyright (c) 1995, 1996, 1997, 1998, 1999, 2000 by
***  The Board of Trustees of the University of Illinois.
***  All rights reserved.
**/

/*
   This class is used to store all of the structural       
   information for a simulation.  It reas in this information 
   from a .psf file, cross checks and obtains some information
   from the Parameters object that is passed in, and then     
   stores all this information for later use.	
*/


#ifndef MOLECULE_H

#define MOLECULE_H

#include "parm.h"
#include "Mindy.h"
#include "Vector.h"
#include "UniqueSet.h"
#include "Hydrogen.h"


class SimParameters;
class Parameters;
template<class Type> class ObjectArena;

struct Donor {
    int  don;
    int  nhyd;
    int  type;
    int* hyd;
};

struct Acceptor {
    int  acc;
    int  nant;
    int  type;
    int* ant;
};

// I disabled PBD otherwise MDForce crashes
// class PDB;
// disabled functions: build_langevin_params,  build_fixed_atoms

// List maintaining the global atom indicies sorted by helix groups.
class Molecule
{
  friend class HBonds;

private:
  Atom *atoms;		   //  Array of atom structures
  ObjectArena<char> *nameArena;
  AtomNameInfo *atomNames; //  Array of atom name info.  Only maintained
                           //  on node 0 for VMD interface
  Bond *bonds;	  	   //  Array of bond structures
  Angle *angles;	   //  Array of angle structures
  Dihedral *dihedrals;	   //  Array of dihedral structures
  Improper *impropers;	   //  Array of improper structures
  Exclusion *exclusions;   //  Array of exclusion structures
  Exclusion *explicitExcl; //  Array of explicit exclusion structures
  UniqueSet<Exclusion> exclusionSet; //  Used for building only

  int    *fixedAtomFlags;    //  1 for fixed, -1 for fixed group, else 0
  double *rigidBondLengths;  //  if H, length to parent or 0. or
                             //  if not H, length between children or 0.
  int *numBondsByAtom;     //  Array of number of bonds each atom posesses
  int *numBondsWithAtom;   //  Array of number of bonds each atom is involved in
  int *numAnglesByAtom;    //  Array of number of angles each atom is involved in
  int *numDihedralsByAtom; //  Array of number of dihedrals each atom is involved in
  int *numImpropersByAtom; //  Array of number of impropers each atom is involved in
  int *donorsByAtom;         // contains 1 for a donor, zero else.
  int *acceptorsByAtom;      // contains 1 for a donor, zero else
    //int *antecedentsByAcceptor; // contains antecedent global index; indexed by Acceptor number

  Donor    *donors;	   //  Array of hydrogen bond donor structures
  Acceptor *acceptors;	   //  Array of hydrogen bond acceptor structures

  int **bondsWithAtom;	  //  List of bonds involving each atom
  int **bondsByAtom;	  //  List of bonds owned by each atom
  int **anglesByAtom;     //  List of angles owned by each atom
  int **dihedralsByAtom;  //  List of dihedrals owned by each atom
  int **impropersByAtom;  //  List of impropers owned by each atom
  int **exclusionsByAtom; //  List of exclusions owned by each atom
  
  int **all_exclusions;
     //  List of all exclusions, including
     //  explicit exclusions and those calculated
     //  from the bonded structure based on the
     //  exclusion policy
  int **onefour_exclusions;
     //  List of 1-4 interactions.  This list is
     //  used only if the exclusion policy is 
     //  scaled1-4 to track 1-4 interactions that
     //  need to be handled differently
  
  
  void read_atoms(FILE *, Parameters *);  //  Read in atom info from .psf
  void read_bonds(FILE *, Parameters *);  //  Read in bond info from .psf
  void read_angles(FILE *, Parameters *); //  Read in angle info from .psf
  void read_dihedrals(FILE *, Parameters *); //  Read in dihedral info from .psf
  void read_impropers(FILE *, Parameters *); //  Read in improper info from .psf
  void read_donors(FILE *);     //  Read in hydrogen bond donors from .psf
  void read_acceptors(FILE *);  //  Read in hydrogen bond acceptors from .psf
  void read_exclusions(FILE *); //  Read in exclusion info from .psf
    
  void build12excl(void);
  void build13excl(void);
  void build14excl(int);
  void stripHGroupExcl(void);
  void build_exclusions();
  
  // analyze the atoms, and determine which are oxygen, etc.
  // this is called after a molecule is sent our (or received in)
  void build_atom_status(void);

  // added during the trasition from 1x to 2
  SimParameters *simParams;
  Parameters *params;

public:
  int numAtoms;		//  Number of atoms 
  int numBonds;		//  Number of bonds
  int numAngles;	//  Number of angles
  int numDihedrals;	//  Number of dihedrals
  int numImpropers;	//  Number of impropers
  int numExclusions;	//  Number of exclusions
  int numTotalExclusions; //  double Total Number of Exclusions // hack
  int numDonors;          // Number of donors in psf file
  int numAcceptors;       // Number of acceptors in psf file
  int numHDonors;          // Number of donors assigned
  int numHAcceptors;       // Number of acceptors assigned

  int numConstraints;	  //  Number of atoms constrained
  int numFixedAtoms;	  //  Number of fixed atoms
  int numHydrogenGroups;  //  Number of hydrogen groups
  int numRigidBonds;	  //  Number of rigid bonds
  int numFixedRigidBonds; //  Number of rigid bonds between fixed atoms
  
  // The following are needed for error checking because we
  // eliminate bonds, etc. which involve only fixed atoms
  int numCalcBonds;	 //  Number of bonds requiring calculation
  int numCalcAngles;	 //  Number of angles requiring calculation
  int numCalcDihedrals;	 //  Number of dihedrals requiring calculation
  int numCalcImpropers;	 //  Number of impropers requiring calculation
  int numCalcExclusions; //  Number of exclusions requiring calculation
  
  //  Number of dihedrals with multiple periodicity
  int numMultipleDihedrals; 
  //  Number of impropers with multiple periodicity
  int numMultipleImpropers; 
  // indexes of "atoms" sorted by hydrogen groups
  HydrogenGroup hydrogenGroup;
  int waterIndex;

  Molecule(SimParameters *, Parameters *param, const char *filename=NULL);

  Molecule(SimParameters *, Parameters *, Ambertoppar *);
  void read_parm(Ambertoppar *);

  ~Molecule();		//  Destructor
  
  void read_psf_file(const char *, Parameters *);
      //  Read in a .psf file given the filename and the parameter
      //  object to use

  void build_lists_by_atom();
      //  Build the list of structures by atom
  
  int is_hydrogen(int);            // return true if atom is hydrogen
  int is_oxygen(int);              // return true if atom is oxygen
  int is_hydrogenGroupParent(int); // return true if atom is group parent
  int is_water(int);               // return true if atom is part of water 

    // Return the atom index of donor num by the donorindex
    int get_donor(int donnum) const {
	if (donnum<0) NAMD_die("ERROR: Negative index in get_donor()!");
	if (donnum>=numDonors) NAMD_die("ERROR: Index too high in get_donor()!");
	return donors[donnum].don;
    };
    
    // Return the atom index of acceptor num by the acceptorindex
    int get_acceptor(int accnum) const { 
	if (accnum<0) NAMD_die("ERROR: Negative index in get_acceptor()!");
	if (accnum>=numHAcceptors) NAMD_die("ERROR: Index too high in get_acceptor()!");
	return acceptors[accnum].acc;
    };
    // Return the number of antecedents of acceptor
    int get_num_antecedents(int accnum) const { 
	if (accnum<0) NAMD_die("ERROR: Negative index in get_antecedent()!");
	if (accnum>=numHAcceptors) NAMD_die("ERROR: Index too high in get_acceptor()!");
	return acceptors[accnum].nant;
    };
    // Return the atom index of acceptor num by the acceptorindex
    void get_antecedents(int accnum, int *nant, int **ant) const { 
	if (accnum<0) NAMD_die("ERROR: Negative index in get_antecedent()!");
	if (accnum>=numHAcceptors) NAMD_die("ERROR: Index too high in get_acceptor()!");
	*nant = acceptors[accnum].nant;
	*ant  = acceptors[accnum].ant;
    };

    // Return the donor type index
    int get_donor_type(int donnum) const {
	return donors[donnum].type;
    };
    
    // Return the acceptor type index
    int get_acceptor_type(int donnum) const {
	return acceptors[donnum].type;
    };
    // Return the donor index if atom is H-bond donor (needs acceptorByAtom list build)
    // else return -1
    int check_donor(int atomnum) const { return donorsByAtom[atomnum]; };
    
    // Return acceptor index if atom is H-bond acceptor (needs acceptorByAtom list build)
    // else return -1
    int check_acceptor(int atomnum) const { return acceptorsByAtom[atomnum]; };

/*
    // Return the antecedent of the acceptor with atom index atomnum
    int check_antecedent(int atomnum) const { 
	int acc=check_acceptor(atomnum);
	if (acc==-1) NAMD_die("ERROR: Antecedent for NON-acceptor requested in check_antecedent()!");
	return antecedentsByAcceptor[acc]; 
    };
*/
    
  int get_groupSize(int);          // return # atoms in (hydrogen) group
  int get_mother_atom(int);        // return mother atom of a hydrogen

  // Return childID's of HydrogenGroup mother atom
  void get_child_atoms(int anum, int* numChildren, int** cID); 

  // Return number of child atoms of HydrogenGroup mother atom
  int get_num_child_atoms(int anum) {
    return hydrogenGroup[atoms[anum].hydrogenList].atomsInGroup-1;
  }

  //  Get the mass of an atom
  double atommass(int anum) const { return(atoms[anum].mass); }
  
  //  Get the charge of an atom
  double atomcharge(int anum) const { return(atoms[anum].charge); }
  
  //  Get the vdw type of an atom
  Index atomvdwtype(int anum) const { return(atoms[anum].vdw_type); }
  
  //  Retrieve a bond structure
  Bond *get_bond(int bnum) const {return (&(bonds[bnum]));}
  
  //  Retrieve an angle structure
  Angle *get_angle(int anum) const {return (&(angles[anum]));}

  //  Retrieve an improper strutcure
  Improper *get_improper(int inum) const {return (&(impropers[inum]));}
  
  //  Retrieve a dihedral structure
  Dihedral *get_dihedral(int dnum) const {return (&(dihedrals[dnum]));}
  
  //  Retrieve an exclusion structure
  Exclusion *get_exclusion(int ex) const {return (&(exclusions[ex]));}
  
  //  Retrieve an exclusion structure
  Exclusion *get_explicit_exclusion(int ex) const {return (&(explicitExcl[ex]));}
  
  //  Retrieve an atom type
  const char *get_atomtype(int anum) const { return(atomNames[anum].atomtype); }

  //  Retrieve atomname
  const char *get_atomname(int anum) const { 
    if (anum>=0) return(atomNames[anum].atomname);
    else NAMD_die("ERROR: Negative index in get_atomname()!");
    return NULL;
  };

  //  Retrieve residue name
  const char *get_resname(int anum) const { 
    if (anum>=0) return(atomNames[anum].resname); 
    else NAMD_die("ERROR: Negative index in get_resname()!");
    return NULL;
  };

  //  Retrieve segment name
  const char *get_segname(int anum) const { 
    if (anum>=0) return(atomNames[anum].segname); 
    else NAMD_die("ERROR: Negative index in get_segname()!");
    return NULL;
};
  
  //  Retrieve residue index
  int get_resid(int anum) const { return(atomNames[anum].resid); };
  
  //  Lookup atom id from segment, residue, and name
  int get_atom_from_name(const char *segid, int resid, const char *aname) const;
  
  //  Lookup number of atoms in residue from segment and residue
  int get_residue_size(const char *segid, int resid) const;
  
  //  Lookup atom id from segment, residue, and index in residue
  int *get_atoms_from_segid_and_residue(const char *segid, int resid, int *index) const;
  
  
  //  The following routines are used to get the list of bonds
  //  for a given atom.  This is used when creating the bond lists
  //  for the force objects
  int *get_bonds_for_atom(int anum)      { return bondsByAtom[anum]; }
  int *get_bonds_with_atom(int anum)     { 
      if (!bondsWithAtom[anum]) NAMD_die("bad bond");
      return bondsWithAtom[anum]; 
  }
  int *get_angles_for_atom(int anum)     { return anglesByAtom[anum]; }
  int *get_dihedrals_for_atom(int anum)  { return dihedralsByAtom[anum]; }
  int *get_impropers_for_atom(int anum)  { return impropersByAtom[anum]; }
  int *get_exclusions_for_atom(int anum) { return exclusionsByAtom[anum]; }

  int get_bond_atom1(int bnum) { return bonds[bnum].atom1; }
  int get_bond_atom2(int bnum) { return bonds[bnum].atom2; }
  int get_num_bonds_for_atom(int anum)     { return numBondsByAtom[anum]; }
  int get_num_bonds_with_atom(int anum)    { return numBondsWithAtom[anum]; }
  int get_num_angles_for_atom(int anum)    { return numAnglesByAtom[anum]; }
  int get_num_dihedrals_for_atom(int anum) { return numDihedralsByAtom[anum]; }
  int get_num_impropers_for_atom(int anum) { return numImpropersByAtom[anum]; }
  int get_num_exclusions()                 { return exclusionSet.size(); }

  void set_fixed_atoms(int *fixedatoms, int numfixed); // Set the fixedAtomsFlags

  //  Check for exclusions, either explicit or bonded.
  //  Inline this funcion since it is called so often
  int checkexcl(int atom1, int atom2) const {
    register int check_int; //  atom whose array we will search
    int other_int;	    //  atom we are looking for
    
    //  We want to search the array of the smaller atom
    if (atom1<atom2) {
      check_int = atom1;
      other_int = atom2;
    }
    else {
      check_int = atom2;
      other_int = atom1;
    }
    
    //  Do the search and return the correct value
    register int *list = all_exclusions[check_int];
    check_int = *list;
    while( check_int != other_int && check_int != -1 ) {
      check_int = *(++list);
    }
    return ( check_int != -1 );
  }
  
  //  Check for 1-4 exclusions.  This is only valid when the
  //  exclusion policy is set to scaled1-4. Inline this function
  //  since it will be called so often
  int check14excl(int atom1, int atom2) const {
    register int check_int;	//  atom whose array we will search
    int other_int;	//  atom we are looking for
    
    //  We want to search the array of the smaller atom
    if (atom1<atom2) {
      check_int = atom1;
      other_int = atom2;
    }
    else {
      check_int = atom2;
      other_int = atom1;
    }
    
    //  Do the search and return the correct value
    register int *list = onefour_exclusions[check_int];
    check_int = *list;
    while( check_int != other_int && check_int != -1 ) {
      check_int = *(++list);
    }
    return ( check_int != -1 );
  }
  
  
  int is_atom_fixed(int atomnum) const {
    return (numFixedAtoms && fixedAtomFlags[atomnum]);
  }
  
  int is_group_fixed(int atomnum) const {
    return (numFixedAtoms && (fixedAtomFlags[atomnum] == -1));
  }
  
  // 0 if not rigid or length to parent, for parent refers to H-H length
  double rigid_bond_length(int atomnum) const {
    return(rigidBondLengths[atomnum]);
  }
  
  void print_atoms(Parameters *);	
  //  Print out list of atoms
  void print_bonds(Parameters *);	
  //  Print out list of bonds
  void print_exclusions(); //  Print out list of exclusions

  void build_donor_list(Parameters* hbparams);
  void build_acceptor_list(Parameters* hbparams);
};

  // Helper for array sorting
  int compare_donors(const void* donor1, const void* donor2);
  int compare_acceptors(const void* acceptor1, const void* acceptor2);


#endif
