/**
***  Copyright (c) 1995, 1996, 1997, 1998, 1999, 2000 by
***  The Board of Trustees of the University of Illinois.
***  All rights reserved.
**/

/*
   The class Molecule is used to hold all of the structural information
   for a simulation.  This information is read in from a .psf file and
   cross checked with the Parameters object passed in.  All of the structural
   information is then stored in arrays for use.
*/

#include "Molecule.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <algorithm>
#include "strlib.h"
#include "ObjectArena.h"
#include "Parameters.h"
//#include "PDB.h"        
#include "Hydrogen.h"
#include "UniqueSetIter.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

#define UnknownAtom      0x00
#define HydrogenAtom     0x01
#define OxygenAtom       0x02

using namespace std;


Molecule::Molecule(SimParameters *simParams, Parameters *param, const char *filename)
{
  this->simParams = simParams;

  atoms=NULL;
  atomNames=NULL;
  bonds=NULL;
  angles=NULL;
  dihedrals=NULL;
  impropers=NULL;
  donors=NULL;
  acceptors=NULL;
  exclusions=NULL;
  explicitExcl=NULL;
  bondsWithAtom=NULL;
  bondsByAtom=NULL;
  anglesByAtom=NULL;
  dihedralsByAtom=NULL;
  impropersByAtom=NULL;
  numBondsByAtom=NULL;
  numBondsWithAtom=NULL;
  numAnglesByAtom=NULL;
  numDihedralsByAtom=NULL;
  numImpropersByAtom=NULL;
  donorsByAtom=NULL;
  acceptorsByAtom=NULL;
  exclusionsByAtom=NULL;
  all_exclusions=NULL;
  onefour_exclusions=NULL;
  fixedAtomFlags=NULL;
  rigidBondLengths=NULL;

  /*  Initialize counts to 0 */
  numAtoms=0;
  numBonds=0;
  numAngles=0;
  numDihedrals=0;
  numImpropers=0;
  numDonors=0;
  numAcceptors=0;
  numHDonors=0;
  numHAcceptors=0;
  numExclusions=0;
  numFixedAtoms=0;
  numRigidBonds=0;
  numFixedRigidBonds=0;
  numMultipleDihedrals=0;
  numMultipleImpropers=0;
  numCalcBonds=0;
  numCalcAngles=0;
  numCalcDihedrals=0;
  numCalcImpropers=0;
  numCalcExclusions=0;

  if (param != NULL && filename != NULL) {
    read_psf_file(filename, param);
  }

  //build_lists_by_atom(); //Must now be invoked manually
}

Molecule::~Molecule()
{
    int i;
    delete [] atoms;
    delete [] bonds;
    delete [] angles;
    delete [] dihedrals;
    delete [] impropers;
    delete [] numAnglesByAtom;
    delete [] numBondsByAtom;
    delete [] numBondsWithAtom;
    delete [] numDihedralsByAtom;
    delete [] numImpropersByAtom;
    delete [] donorsByAtom;
    delete [] acceptorsByAtom;
    delete [] exclusions;
    delete [] explicitExcl;
    for (i=0; i<numAtoms; i++) {
      delete [] bondsByAtom[i];
      delete [] bondsWithAtom[i];
      delete [] anglesByAtom[i];
      delete [] dihedralsByAtom[i];
      delete [] impropersByAtom[i];
      delete [] exclusionsByAtom[i];
      delete [] all_exclusions[i];
      delete [] onefour_exclusions[i];
      delete [] atomNames[i].segname;
      delete [] atomNames[i].resname;
      delete [] atomNames[i].atomname;
      delete [] atomNames[i].atomtype;
    }
    delete [] bondsByAtom;
    delete [] bondsWithAtom;
    delete [] anglesByAtom;
    delete [] dihedralsByAtom;
    delete [] impropersByAtom;
    delete [] exclusionsByAtom;
    delete [] all_exclusions;
    delete [] onefour_exclusions;
    delete [] atomNames;
    delete [] fixedAtomFlags;
    delete [] rigidBondLengths;
    for (i=0; i<numHDonors; i++) {
      delete [] donors[i].hyd;
    }
    for (i=0; i<numHAcceptors; i++) {
      delete [] acceptors[i].ant;
    }
    delete [] donors;
    delete [] acceptors;
}

void Molecule::read_psf_file(const char *fname, Parameters *params)

{
  char err_msg[512];  //  Error message for NAMD_die
  char buffer[512];  //  Buffer for file reading
  int i;      //  Loop counter
  int NumTitle;    //  Number of Title lines in .psf file
  FILE *psf_file;    //  pointer to .psf file
  int ret_code;    //  ret_code from NAMD_read_line calls

  /* Try and open the .psf file           */
  if ( (psf_file = fopen(fname, "r")) == NULL)
  {
    sprintf(err_msg, "UNABLE TO OPEN .psf FILE %s", fname);
    NAMD_die(err_msg);
  }

  /*  Read till we have the first non-blank line of file    */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to see if we dropped out of the loop because of a     */
  /*  read error.  This shouldn't happen unless the file is empty */
  if (ret_code!=0)
  {
    sprintf(err_msg, "EMPTY .psf FILE %s", fname);
    NAMD_die(err_msg);
  }

  /*  The first non-blank line should contain the word "psf".    */
  /*  If we can't find it, die.               */
  if (!NAMD_find_word(buffer, "psf"))
  {
    sprintf(err_msg, "UNABLE TO FIND \"PSF\" STRING IN PSF FILE %s",
       fname);
    NAMD_die(err_msg);
  }

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to see if we dropped out of the loop because of a     */
  /*  read error.  This shouldn't happen unless there is nothing  */
  /*  but the PSF line in the file        */
  if (ret_code!=0)
  {
    sprintf(err_msg, "MISSING EVERYTHING BUT PSF FROM %s", fname);
    NAMD_die(err_msg);
  }

  /*  This line should have the word "NTITLE" in it specifying    */
  /*  how many title lines there are        */
  if (!NAMD_find_word(buffer, "NTITLE"))
  {
    sprintf(err_msg,"CAN NOT FIND \"NTITLE\" STRING IN PSF FILE %s",
       fname);
    NAMD_die(err_msg);
  }

  sscanf(buffer, "%d", &NumTitle);

  /*  Now skip the next NTITLE non-blank lines and then read in the*/
  /*  line which should contain NATOM        */
  i=0;

  while ( ((ret_code=NAMD_read_line(psf_file, buffer)) == 0) && 
    (i<NumTitle) )
  {
    if (!NAMD_blank_string(buffer))
      i++;
  }

  /*  Make sure we didn't exit because of a read error    */
  if (ret_code!=0)
  {
    sprintf(err_msg, "FOUND EOF INSTEAD OF NATOM IN PSF FILE %s", 
       fname);
    NAMD_die(err_msg);
  }

  while (NAMD_blank_string(buffer))
  {
    NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we have the line we want      */
  if (!NAMD_find_word(buffer, "NATOM"))
  {
    sprintf(err_msg, "DIDN'T FIND \"NATOM\" IN PSF FILE %s",
       fname);
    NAMD_die(err_msg);
  }

  /*  Read in the number of atoms, and then the atoms themselves  */
  sscanf(buffer, "%d", &numAtoms);

  read_atoms(psf_file, params);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    NAMD_die("EOF ENCOUNTERED LOOKING FOR NBONDS IN PSF");
  }

  /*  Look for the string "NBOND"          */
  if (!NAMD_find_word(buffer, "NBOND"))
  {
    NAMD_die("DID NOT FIND NBOND AFTER ATOM LIST IN PSF");
  }

  /*  Read in the number of bonds and then the bonds themselves  */
  sscanf(buffer, "%d", &numBonds);

  if (numBonds)
    read_bonds(psf_file, params);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    NAMD_die("EOF ENCOUNTERED LOOKING FOR NTHETA IN PSF");
  }

  /*  Look for the string "NTHETA"        */
  if (!NAMD_find_word(buffer, "NTHETA"))
  {
    NAMD_die("DID NOT FIND NTHETA AFTER BOND LIST IN PSF");
  }

  /*  Read in the number of angles and then the angles themselves */
  sscanf(buffer, "%d", &numAngles);

  if (numAngles)
    read_angles(psf_file, params);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    NAMD_die("EOF ENCOUNTERED LOOKING FOR NPHI IN PSF");
  }

  /*  Look for the string "NPHI"          */
  if (!NAMD_find_word(buffer, "NPHI"))
  {
    NAMD_die("DID NOT FIND NPHI AFTER ANGLE LIST IN PSF");
  }

  /*  Read in the number of dihedrals and then the dihedrals      */
  sscanf(buffer, "%d", &numDihedrals);

  if (numDihedrals)
    read_dihedrals(psf_file, params);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    NAMD_die("EOF ENCOUNTERED LOOKING FOR NIMPHI IN PSF");
  }

  /*  Look for the string "NIMPHI"        */
  if (!NAMD_find_word(buffer, "NIMPHI"))
  {
    NAMD_die("DID NOT FIND NIMPHI AFTER ATOM LIST IN PSF");
  }

  /*  Read in the number of Impropers and then the impropers  */
  sscanf(buffer, "%d", &numImpropers);

  if (numImpropers)
    read_impropers(psf_file, params);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    NAMD_die("EOF ENCOUNTERED LOOKING FOR NDON IN PSF");
  }

  /*  Look for the string "NDON"        */
  if (!NAMD_find_word(buffer, "NDON"))
  {
    NAMD_die("DID NOT FIND NDON AFTER ATOM LIST IN PSF");
  }

  /*  Read in the number of hydrogen bond donors and then the donors */
  sscanf(buffer, "%d", &numDonors);

  if (numDonors)
    read_donors(psf_file);

  /*  Read until we find the next non-blank line      */
  ret_code = NAMD_read_line(psf_file, buffer);
    cerr << buffer;

  while ( (ret_code==0) && (NAMD_blank_string(buffer)) )
  {
    ret_code = NAMD_read_line(psf_file, buffer);
  }

  /*  Check to make sure we didn't hit the EOF      */
  if (ret_code != 0)
  {
    NAMD_die("EOF ENCOUNTERED LOOKING FOR NACC IN PSF");
  }

  /*  Look for the string "NACC"        */
  if (!NAMD_find_word(buffer, "NACC"))
  {
    NAMD_die("DID NOT FIND NACC AFTER ATOM LIST IN PSF");
  }

  /*  Read in the number of hydrogen bond donors and then the donors */
  sscanf(buffer, "%d", &numAcceptors);

  if (numAcceptors)
    read_acceptors(psf_file);

  /*  look for the explicit non-bonded exclusion section.     */
  while (!NAMD_find_word(buffer, "NNB"))
  {
    ret_code = NAMD_read_line(psf_file, buffer);

    if (ret_code != 0)
    {
      NAMD_die("EOF ENCOUNTERED LOOKING FOR NNB IN PSF FILE");
    }
  }

  /*  Read in the number of exclusions and then the exclusions    */
  sscanf(buffer, "%d", &numExclusions);

  if (numExclusions)
    read_exclusions(psf_file);

  /*  Close the .psf file.  There is a Group section in the .psf  */
  /*  file after the NNB section, but currently, NAMD does not    */
  /*  use this section for anything.        */
  fclose(psf_file);

  //  analyze the data and find the status of each atom
  build_atom_status();

  return;
}
/*      END OF FUNCTION read_psf_file      */

/************************************************************************/
/*                  */
/*        FUNCTION read_atoms      */
/*                  */
/*   INPUTS:                */
/*  fd - file pointer to the .psf file        */
/*  params - Parameters object to use for parameters    */
/*                  */
/*  this function reads in the Atoms section of the .psf file.      */
/*   This section consists of numAtoms lines that are of the form:  */
/*     <atom#> <mol> <seg#> <res> <atomname> <atomtype> <charge> <mass> */
/*   Each line is read into the appropriate entry in the atoms array.   */
/*   The parameters object is then used to determine the vdW constants  */
/*   for this atom.              */
/*                  */
/************************************************************************/

void Molecule::read_atoms(FILE *fd, Parameters *params)

{
  char buffer[512];  // Buffer for reading from file
  int atom_number=0;  // Atom number 
  int last_atom_number=0; // Last atom number, used to assure
        // atoms are in order
  char segment_name[11];   // Segment name
  char residue_number[11]; // Residue number
  char residue_name[11];   // Residue name
  char atom_name[11];      // Atom name
  char atom_type[11];      // Atom type
  double charge;           // Charge for the current atom
  double mass;             // Mass for the current atom
  int read_count;          // Number of fields read by sscanf

  /*  Allocate the atom arrays          */
  atoms     = new Atom[numAtoms];
  atomNames = new AtomNameInfo[numAtoms];
  hydrogenGroup.resize(0);

  if (atoms == NULL || atomNames == NULL )
  {
    NAMD_die("memory allocation failed in Molecule::read_atoms");
  }

  /*  Loop and read in numAtoms atom lines.      */
  while (atom_number < numAtoms)
  {
    /*  Get the line from the file        */
    NAMD_read_line(fd, buffer);

    /*  If its blank or a comment, skip it      */
    if ( (NAMD_blank_string(buffer)) || (buffer[0] == '!') )
      continue;

    /*  Parse up the line          */
    read_count=sscanf(buffer, "%d %s %s %s %s %s %lf %lf",
       &atom_number, segment_name, residue_number,
       residue_name, atom_name, atom_type, &charge, &mass);

    /*  Check to make sure we found what we were expecting  */
    if (read_count != 8)
    {
      char err_msg[128];

      sprintf(err_msg, "BAD ATOM LINE FORMAT IN PSF FILE IN ATOM LINE %d\nLINE=%s",
         last_atom_number+1, buffer);
      NAMD_die(err_msg);
    }

    /*  Make sure the atoms were in sequence    */
    if (atom_number != last_atom_number+1)
    {
      char err_msg[128];

      sprintf(err_msg, "ATOM NUMBERS OUT OF ORDER AT ATOM #%d OF PSF FILE",
         last_atom_number+1);
      NAMD_die(err_msg);
    }

    last_atom_number++;

    /*  Dynamically allocate strings for atom name, atom    */
    /*  type, etc so that we only allocate as much space    */
    /*  for these strings as we really need      */
    int seglength = strlen(segment_name)+1;
    int reslength = strlen(residue_name)+1;
    int namelength = strlen(atom_name)+1;
    int typelength = strlen(atom_type)+1;

    atomNames[atom_number-1].resid = atoi(residue_number);
    atomNames[atom_number-1].resname = new char[reslength];
    atomNames[atom_number-1].segname = new char[seglength];
    atomNames[atom_number-1].atomname = new char[namelength];
    atomNames[atom_number-1].atomtype = new char[typelength];
  
    if (atomNames[atom_number-1].resname == NULL)
      NAMD_die("memory allocation failed in Molecule::read_atoms");
    if (atomNames[atom_number-1].segname == NULL)
      NAMD_die("memory allocation failed in Molecule::read_atoms");

    /*  Put the values from this atom into the atoms array  */
    strcpy(atomNames[atom_number-1].segname, segment_name);
    strcpy(atomNames[atom_number-1].resname, residue_name);
    strcpy(atomNames[atom_number-1].atomname, atom_name);
    strcpy(atomNames[atom_number-1].atomtype, atom_type);
    atoms[atom_number-1].mass = mass;
    atoms[atom_number-1].charge = charge;
    atoms[atom_number-1].status = UnknownAtom;

    /*  Determine the type of the atom (H or O) */
    if (atoms[atom_number-1].mass <=3.5) {
      atoms[atom_number-1].status |= HydrogenAtom;
    } else if ((atomNames[atom_number-1].atomname[0] == 'O') && 
         (atoms[atom_number-1].mass >= 14.0) && 
         (atoms[atom_number-1].mass <= 18.0)) {
      atoms[atom_number-1].status |= OxygenAtom;
    }

    /*  Look up the vdw constants for this atom    */
    params->assign_vdw_index(atomNames[atom_number-1].atomtype, 
       &(atoms[atom_number-1]));
        }

  return;
}
/*      END OF FUNCTION read_atoms      */

/************************************************************************/
/*                  */
/*      FUNCTION read_bonds        */
/*                  */
/*  read_bonds reads in the bond section of the .psf file.  This    */
/*  section contains a list of pairs of numbers where each pair is      */
/*  represents two atoms that are bonded together.  Each atom pair is   */
/*  read in.  Then that parameter object is queried to determine the    */
/*  force constant and rest distance for the bond.      */
/*                  */
/************************************************************************/

void Molecule::read_bonds(FILE *fd, Parameters *params)

{
  int atom_nums[2];  // Atom indexes for the bonded atoms
  char atom1name[11];  // Atom type for atom #1
  char atom2name[11];  // Atom type for atom #2
  register int j;      // Loop counter
  int num_read=0;    // Number of bonds read so far
  int origNumBonds = numBonds;   // number of bonds in file header

  /*  Allocate the array to hold the bonds      */
  bonds=new Bond[numBonds];

  if (bonds == NULL)
  {
    NAMD_die("memory allocations failed in Molecule::read_bonds");
  }

  /*  Loop through and read in all the bonds      */
  while (num_read < numBonds)
  {
    /*  Loop and read in the two atom indexes    */
    for (j=0; j<2; j++)
    {
      /*  Read the atom number from the file.         */
      /*  Subtract 1 to convert the index from the    */
      /*  1 to NumAtoms used in the file to the       */
      /*  0 to NumAtoms-1 that we need    */
      atom_nums[j]=NAMD_read_int(fd, "BONDS")-1;

      /*  Check to make sure the index isn't too big  */
      if (atom_nums[j] >= numAtoms)
      {
        char err_msg[128];

        sprintf(err_msg, "BOND INDEX %d GREATER THAN NATOM %d IN BOND # %d IN PSF FILE", atom_nums[j]+1, numAtoms, num_read+1);
        NAMD_die(err_msg);
      }
    }

    /*  Get the atom type for the two atoms.  When we query */
    /*  the parameter object, we need to send the atom type */
    /*  that is alphabetically first as atom 1.    */
    if (strcasecmp(atomNames[atom_nums[0]].atomtype, 
         atomNames[atom_nums[1]].atomtype) < 0)
    {
      strcpy(atom1name, atomNames[atom_nums[0]].atomtype);
      strcpy(atom2name, atomNames[atom_nums[1]].atomtype);
    }
    else
    {
      strcpy(atom2name, atomNames[atom_nums[0]].atomtype);
      strcpy(atom1name, atomNames[atom_nums[1]].atomtype);
    }

    /*  Query the parameter object for the constants for    */
    /*  this bond            */
    Bond *b = &(bonds[num_read]);
    params->assign_bond_index(atom1name, atom2name, b);

    /*  Assign the atom indexes to the array element  */
    b->atom1=atom_nums[0];
    b->atom2=atom_nums[1];

    /*  Make sure this isn't a fake bond meant for shake in x-plor.  */
    double k, x0;
    params->get_bond_params(&k,&x0,b->bond_type);
    if ( k == 0. ) --numBonds;  // fake bond
    else ++num_read;  // real bond
  }

  /*  Tell user about our subterfuge  */
  if ( numBonds != origNumBonds ) {
    cout <<   "Ignored " << origNumBonds - numBonds <<
            " bonds with zero force constants.\n" << endl;
    cout <<  
	"Will get H-H distance in rigid H2O from H-O-H angle.\n" << endl;
  }

  return;
}
/*      END OF FUNCTION read_bonds      */

/************************************************************************/
/*                  */
/*      FUNCTION read_angles        */
/*                  */
/*   INPUTS:                */
/*  fd - File descriptor for .psf file        */
/*  params - Parameters object to query for parameters    */
/*                  */
/*  read_angles reads the angle parameters from the .psf file.      */
/*   This section of the .psf file consists of a list of triplets of    */
/*   atom indexes.  Each triplet represents three atoms connected via   */
/*   an angle bond.  The parameter object is queried to obtain the      */
/*   constants for each bond.            */
/*                  */
/************************************************************************/

void Molecule::read_angles(FILE *fd, Parameters *params)

{
  int atom_nums[3];  //  Atom numbers for the three atoms
  char atom1name[11];  //  Atom type for atom 1
  char atom2name[11];  //  Atom type for atom 2
  char atom3name[11];  //  Atom type for atom 3
  register int j;      //  Loop counter
  int num_read=0;    //  Number of angles read so far
  int origNumAngles = numAngles;  // Number of angles in file
  /*  Alloc the array of angles          */
  angles=new Angle[numAngles];

  if (angles == NULL)
  {
    NAMD_die("memory allocation failed in Molecule::read_angles");
  }

  /*  Loop through and read all the angles      */
  while (num_read < numAngles)
  {
    /*  Loop through the 3 atom indexes in the current angle*/
    for (j=0; j<3; j++)
    {
      /*  Read the atom number from the file.         */
      /*  Subtract 1 to convert the index from the    */
      /*  1 to NumAtoms used in the file to the       */
      /*  0 to NumAtoms-1 that we need    */
      atom_nums[j]=NAMD_read_int(fd, "ANGLES")-1;

      /*  Check to make sure the atom index doesn't   */
      /*  exceed the Number of Atoms      */
      if (atom_nums[j] >= numAtoms)
      {
        char err_msg[128];

        sprintf(err_msg, "ANGLES INDEX %d GREATER THAN NATOM %d IN ANGLES # %d IN PSF FILE", atom_nums[j]+1, numAtoms, num_read+1);
        NAMD_die(err_msg);
      }
    }

    /*  Place the bond name that is alphabetically first  */
    /*  in the atom1name.  This is OK since the order of    */
    /*  atom1 and atom3 are interchangable.  And to search  */
    /*  the tree of angle parameters, we need the order     */
    /*  to be predictable.          */
    if (strcasecmp(atomNames[atom_nums[0]].atomtype, 
         atomNames[atom_nums[2]].atomtype) < 0)
    {
      strcpy(atom1name, atomNames[atom_nums[0]].atomtype);
      strcpy(atom2name, atomNames[atom_nums[1]].atomtype);
      strcpy(atom3name, atomNames[atom_nums[2]].atomtype);
    }
    else
    {
      strcpy(atom1name, atomNames[atom_nums[2]].atomtype);
      strcpy(atom2name, atomNames[atom_nums[1]].atomtype);
      strcpy(atom3name, atomNames[atom_nums[0]].atomtype);
    }

    /*  Get the constant values for this bond from the  */
    /*  parameter object          */
    params->assign_angle_index(atom1name, atom2name, 
       atom3name, &(angles[num_read]));

    /*  Assign the three atom indices      */
    angles[num_read].atom1=atom_nums[0];
    angles[num_read].atom2=atom_nums[1];
    angles[num_read].atom3=atom_nums[2];

    /*  Make sure this isn't a fake angle meant for shake in x-plor.  */
    double k, t0, k_ub, r_ub;
    params->get_angle_params(&k,&t0,&k_ub,&r_ub,angles[num_read].angle_type);
    if ( k == 0. && k_ub == 0. ) --numAngles;  // fake angle
    else ++num_read;  // real angle
  }

  /*  Tell user about our subterfuge  */
  if ( numAngles != origNumAngles ) {
    cout <<   "Ignored " << origNumAngles - numAngles <<
            " angles with zero force constants.\n" << endl;
  }

  return;
}
/*      END OF FUNCTION read_angles      */

/************************************************************************/
/*                  */
/*        FUNCTION read_dihedrals      */
/*                  */
/*   INPUTS:                */
/*  fd - file descriptor for the .psf file        */
/*  params - pointer to parameter object        */
/*                  */
/*  read_dihedreals reads the dihedral section of the .psf file.    */
/*   This section of the file contains a list of quartets of atom       */
/*   numbers.  Each quartet represents a group of atoms that form a     */
/*   dihedral bond.              */
/*                  */
/************************************************************************/

void Molecule::read_dihedrals(FILE *fd, Parameters *params)

{
  int atom_nums[4];  // The 4 atom indexes
  int last_atom_nums[4];  // Atom numbers from previous bond
  char atom1name[11];  // Atom type for atom 1
  char atom2name[11];  // Atom type for atom 2
  char atom3name[11];  // Atom type for atom 3
  char atom4name[11];  // Atom type for atom 4
  register int j;      // loop counter
  int num_read=0;    // number of dihedrals read so far
  int multiplicity=1;  // multiplicity of the current bond
  int duplicate_bond;  // Is this a duplicate of the last bond
  int num_unique=0;   // Number of unique dihedral bonds

  //  Initialize the array used to check for duplicate dihedrals
  for (j=0; j<4; j++)
    last_atom_nums[j] = -1;

  /*  Allocate an array to hold the Dihedrals      */
  dihedrals = new Dihedral[numDihedrals];

  if (dihedrals == NULL)
  {
    NAMD_die("memory allocation failed in Molecule::read_dihedrals");
  }

  /*  Loop through and read all the dihedrals      */
  while (num_read < numDihedrals)
  {
    duplicate_bond = 1;

    /*  Loop through and read the 4 indexes for this bond   */
    for (j=0; j<4; j++)
    {
      /*  Read the atom number from the file.         */
      /*  Subtract 1 to convert the index from the    */
      /*  1 to NumAtoms used in the file to the       */
      /*  0 to NumAtoms-1 that we need    */
      atom_nums[j]=NAMD_read_int(fd, "DIHEDRALS")-1;

      /*  Check for an atom index that is too large  */
      if (atom_nums[j] >= numAtoms)
      {
        char err_msg[128];

        sprintf(err_msg, "DIHEDRALS INDEX %d GREATER THAN NATOM %d IN DIHEDRALS # %d IN PSF FILE", atom_nums[j]+1, numAtoms, num_read+1);
        NAMD_die(err_msg);
      }

      //  Check to see if this atom matches the last bond
      if (atom_nums[j] != last_atom_nums[j])
      {
        duplicate_bond = 0;
      }

      last_atom_nums[j] = atom_nums[j];
    }

    /*  Get the atom types for the 4 atoms so we can look  */
    /*  up the constants in the parameter object    */
    strcpy(atom1name, atomNames[atom_nums[0]].atomtype);
    strcpy(atom2name, atomNames[atom_nums[1]].atomtype);
    strcpy(atom3name, atomNames[atom_nums[2]].atomtype);
    strcpy(atom4name, atomNames[atom_nums[3]].atomtype);

    //  Check to see if this is really a new bond or just
    //  a repeat of the last one
    if (duplicate_bond)
    {
      //  This is a duplicate, so increase the multiplicity
      multiplicity++;

      if (multiplicity == 2)
      {
        numMultipleDihedrals++;
      }
    }
    else
    {
      multiplicity=1;
      num_unique++;
    }

    /*  Get the constants for this dihedral bond    */
    params->assign_dihedral_index(atom1name, atom2name, 
       atom3name, atom4name, &(dihedrals[num_unique-1]),
       multiplicity);

    /*  Assign the atom indexes        */
    dihedrals[num_unique-1].atom1=atom_nums[0];
    dihedrals[num_unique-1].atom2=atom_nums[1];
    dihedrals[num_unique-1].atom3=atom_nums[2];
    dihedrals[num_unique-1].atom4=atom_nums[3];

    num_read++;
  }

  numDihedrals = num_unique;

  return;
}
/*      END OF FUNCTION read_dihedral      */

/************************************************************************/
/*                  */
/*        FUNCTION read_impropers      */
/*                  */
/*   INPUTS:                */
/*  fd - file descriptor for .psf file        */
/*  params - parameter object          */
/*                  */
/*  read_impropers reads the improper section of the .psf file.  */
/*   This section is identical to the dihedral section in that it is    */
/*   made up of a list of quartets of atom indexes that define the      */
/*   atoms that are bonded together.          */
/*                  */
/************************************************************************/

void Molecule::read_impropers(FILE *fd, Parameters *params)

{
  int atom_nums[4];     //  Atom indexes for the 4 atoms
  int last_atom_nums[4];//  Atom indexes from previous bond
  char atom1name[11];   //  Atom type for atom 1
  char atom2name[11];   //  Atom type for atom 2
  char atom3name[11];   //  Atom type for atom 3
  char atom4name[11];   //  Atom type for atom 4
  register int j;       //  Loop counter
  int num_read=0;       //  Number of impropers read so far
  int multiplicity=1;   // multiplicity of the current bond
  int duplicate_bond;   // Is this a duplicate of the last bond
  int num_unique=0;     // Number of unique dihedral bonds

  //  Initialize the array used to look for duplicate improper
  //  entries.  Set them all to -1 so we know nothing will match
  for (j=0; j<4; j++)
    last_atom_nums[j] = -1;

  /*  Allocate the array to hold the impropers      */
  impropers=new Improper[numImpropers];

  if (impropers == NULL)
  {
    NAMD_die("memory allocation failed in Molecule::read_impropers");
  }

  /*  Loop through and read all the impropers      */
  while (num_read < numImpropers)
  {
    duplicate_bond = 1;

    /*  Loop through the 4 indexes for this improper  */
    for (j=0; j<4; j++)
    {
      /*  Read the atom number from the file.         */
      /*  Subtract 1 to convert the index from the    */
      /*  1 to NumAtoms used in the file to the       */
      /*  0 to NumAtoms-1 that we need    */
      atom_nums[j]=NAMD_read_int(fd, "IMPROPERS")-1;

      /*  Check to make sure the index isn't too big  */
      if (atom_nums[j] >= numAtoms)
      {
        char err_msg[128];

        sprintf(err_msg, "IMPROPERS INDEX %d GREATER THAN NATOM %d IN IMPROPERS # %d IN PSF FILE", atom_nums[j]+1, numAtoms, num_read+1);
        NAMD_die(err_msg);
      }

      if (atom_nums[j] != last_atom_nums[j])
      {
        duplicate_bond = 0;
      }

      last_atom_nums[j] = atom_nums[j];
    }

    /*  Get the atom types so we can look up the parameters */
    strcpy(atom1name, atomNames[atom_nums[0]].atomtype);
    strcpy(atom2name, atomNames[atom_nums[1]].atomtype);
    strcpy(atom3name, atomNames[atom_nums[2]].atomtype);
    strcpy(atom4name, atomNames[atom_nums[3]].atomtype);

    //  Check to see if this is a duplicate improper
    if (duplicate_bond)
    {
      //  This is a duplicate improper.  So we don't
      //  really count this entry, we just update
      //  the parameters object
      multiplicity++;

      if (multiplicity == 2)
      {
        //  Count the number of multiples.
        numMultipleImpropers++;
      }
    }
    else
    {
      //  Not a duplicate
      multiplicity = 1;
      num_unique++;
    }

    /*  Look up the constants for this bond      */
    params->assign_improper_index(atom1name, atom2name, 
       atom3name, atom4name, &(impropers[num_unique-1]),
       multiplicity);

    /*  Assign the atom indexes        */
    impropers[num_unique-1].atom1=atom_nums[0];
    impropers[num_unique-1].atom2=atom_nums[1];
    impropers[num_unique-1].atom3=atom_nums[2];
    impropers[num_unique-1].atom4=atom_nums[3];

    num_read++;
  }

  //  Now reset the numImpropers value to the number of UNIQUE
  //  impropers.  Sure, we waste a few entries in the improper_array
  //  on the master node, but it is very little space . . .
  numImpropers = num_unique;

  return;
}
/*      END OF FUNCTION read_impropers      */

// Read the donors.

void Molecule::read_donors(FILE *fd) {
  int atom_nums[2];  // Atom indexes for the bonded atoms
  register int i,j;      // Loop counter
  int num_read=0;    // Number of bonds read so far

  /*  Allocate the array to hold the donors      */
  Donor* tmpdonors=new Donor[numDonors];

  if (tmpdonors == NULL)
  {
    NAMD_die("memory allocations failed in Molecule::read_donors");
  }

  /*  Loop through and read in all the bonds      */
  while (num_read < numDonors)
  {
    /*  Loop and read in the two atom indexes    */
    for (j=0; j<2; j++)
    {
      /*  Read the atom number from the file.         */
      /*  Subtract 1 to convert the index from the    */
      /*  1 to NumAtoms used in the file to the       */
      /*  0 to NumAtoms-1 that we need    */
      atom_nums[j]=NAMD_read_int(fd, "DONORS")-1;

      /*  Check to make sure the index isn't too big  */
      if (atom_nums[j] >= numAtoms)
      {
        char err_msg[128];
        sprintf(err_msg, "DONOR INDEX %d GREATER THAN NATOM %d IN DONOR # %d IN PSF FILE", atom_nums[j]+1, numAtoms, num_read+1);
        NAMD_die(err_msg);
      }
    }

    /*  Assign the atom indexes to the array element  */
    tmpdonors[num_read].don=atom_nums[0];
    /* For now, before we sort the array, store the hydrogen index in nhyd */
    tmpdonors[num_read].nhyd=atom_nums[1];
    ++num_read;  
  }

  qsort (tmpdonors, num_read, sizeof(Donor), compare_donors);
  /*
  for (i=0; i<num_read; i++) {
      cerr << "don/hyd " << tmpdonors[i].don << "  " << tmpdonors[i].nhyd << endl;
  }
  */

  int n=0;
  for (i=0; i<num_read; i++) {
      tmpdonors[n].don = tmpdonors[i].don;
      int curdon = tmpdonors[n].don;  
      //cerr << i << ": don " << tmpdonors[n].don << endl;

      // Count the hydrogens
      int nhyd=0;
      while (tmpdonors[i].don==curdon) { i++; nhyd++; }
      tmpdonors[n].hyd = new int[nhyd];
      //cerr << curdon << "  nhyd=" << nhyd << endl;
      i -= nhyd; // Rewind
      
      // Assign the indices
      int j;
      for (j=0; j<nhyd; j++) {
	  tmpdonors[n].hyd[j] = tmpdonors[i++].nhyd;
	  //cerr << "  hyd " << tmpdonors[n].hyd[j] << endl;
      }
      tmpdonors[n].nhyd = nhyd;
      n++;
      i--;
  }
  numHDonors = n;
  cerr << "numHDonors="<<numHDonors<<endl;

  donors=new Donor[numHDonors];
  if (donors == NULL)
  {
    NAMD_die("memory allocations failed in Molecule::read_donors");
  }

  for (i=0; i<numHDonors; i++) {
      donors[i]=tmpdonors[i];
  }
  delete [] tmpdonors;
}

void Molecule::read_acceptors(FILE *fd) {

  int atom_nums[2];  // Atom indexes for the bonded atoms
  register int i,j;      // Loop counter
  int num_read=0;    // Number of bonds read so far

  /*  Allocate the array to hold the acceptors      */
  Acceptor* tmpacceptors=new Acceptor[numAcceptors];

  if (tmpacceptors == NULL)
  {
    NAMD_die("memory allocations failed in Molecule::read_acceptors");
  }

  /*  Loop through and read in all the bonds      */
  while (num_read < numAcceptors)
  {
    /*  Loop and read in the two atom indexes    */
    for (j=0; j<2; j++)
    {
      /*  Read the atom number from the file.         */
      /*  Subtract 1 to convert the index from the    */
      /*  1 to NumAtoms used in the file to the       */
      /*  0 to NumAtoms-1 that we need    */
      atom_nums[j]=NAMD_read_int(fd, "ACCEPTORS")-1;

      /*  Check to make sure the index isn't too big  */
      if (atom_nums[j] >= numAtoms)
      {
        char err_msg[128];
        sprintf(err_msg, "ACCEPTOR INDEX %d GREATER THAN NATOM %d IN ACCEPTOR # %d IN PSF FILE", atom_nums[j]+1, numAtoms, num_read+1);
        NAMD_die(err_msg);
      }
    }

    /*  Assign the atom indexes to the array element  */
    tmpacceptors[num_read].acc=atom_nums[0];
    /* For now, before we sort the array, store the antecedent index in nant */
    tmpacceptors[num_read].nant=atom_nums[1];
    ++num_read;  
  }

  qsort (tmpacceptors, num_read, sizeof(Acceptor), compare_acceptors);
  /*
    for (int i=0; i<num_read; i++) {
      cerr << "acc/ant " << tmpacceptors[i].acc << "  " << tmpacceptors[i].nant << endl;
    }
  */

  int n=0;
  for (i=0; i<num_read; i++) {
      tmpacceptors[n].acc = tmpacceptors[i].acc;
      int curacc = tmpacceptors[n].acc;  
      //cerr << i << ": acc " << tmpacceptors[n].acc << endl;

      // Count the antecedents
      int nant=0;
      while (tmpacceptors[i].acc==curacc) { i++; nant++; } //i+=nant
      tmpacceptors[n].ant = new int[nant];
      //cerr << curacc << "  nant=" << nant << endl;
      i -= nant; // Rewind


      // Assign the indices
      for (int j=0; j<nant; j++) {
	  tmpacceptors[n].ant[j] = tmpacceptors[i++].nant;
	  //cerr << "  ant " << tmpacceptors[n].ant[j] << endl;
      }
      tmpacceptors[n].nant=nant;
      n++;
      i--;
  }
  numHAcceptors = n;
  cerr << "numHAcceptors="<<numHAcceptors<<endl;

  acceptors=new Acceptor[numHAcceptors];
  if (acceptors == NULL)
  {
    NAMD_die("memory allocations failed in Molecule::read_acceptors");
  }
  for (i=0; i<numHAcceptors; i++) {
      acceptors[i]=tmpacceptors[i];
  }
  delete [] tmpacceptors;
}

int compare_donors(const void *d1, const void *d2) {
    const Donor *donor1 = (const Donor*) d1;
    const Donor *donor2 = (const Donor*) d2;
    if      (donor1->don >  donor2->don) { return 1; }
    else if (donor1->don == donor2->don) { return 0; }
    else    { return -1; }
}
int compare_acceptors(const void *d1, const void *d2) {
    const Acceptor *acceptor1 = (const Acceptor*) d1;
    const Acceptor *acceptor2 = (const Acceptor*) d2;
    if      (acceptor1->acc >  acceptor2->acc) { return 1; }
    else if (acceptor1->acc == acceptor2->acc) { return 0; }
    else    { return -1; }
}



/************************************************************************/
/*                  */
/*      FUNCTION read_exclusions      */
/*                  */
/*   INPUTS:                */
/*  fd - file descriptor for .psf file        */
/*                  */
/*  read_exclusions reads in the explicit non-bonded exclusions     */
/*  from the .psf file.  This section is a little funky, so hang on.    */
/*  Ok, first there is a list of atom indexes that is NumExclusions     */
/*  long.  These are in some sense the atoms that will be exlcuded.     */
/*  Following this list is a list of NumAtoms length that is a list     */
/*  of indexes into the list of excluded atoms.  So an example.  Suppose*/
/*  we have a 5 atom simulation with 3 explicit exclusions.  The .psf   */
/*  file could look like:            */
/*                  */
/*  3!NNB                */
/*  3 4 5                */
/*  0 1 3 3 3              */
/*                  */
/*  This would mean that atom 1 has no explicit exclusions.  Atom 2     */
/*  has an explicit exclusion with atom 3.  Atom 3 has an explicit      */
/*  exclusion with atoms 4 AND 5.  And atoms 4 and 5 have no explicit   */
/*  exclusions.  Got it!?!  I'm not sure who dreamed this up . . .      */
/*                  */
/************************************************************************/

void Molecule::read_exclusions(FILE *fd)

{
  int *exclusion_atoms;    //  Array of indexes of excluded atoms
  register int num_read=0; //  Number fo exclusions read in
  int current_index;   //  Current index value
  int last_index;      //  the previous index value
  register int insert_index=0;  //  index of where we are in exlcusions array

  /*  Allocate the array of exclusion structures and the array of */
  /*  exlcuded atom indexes          */
  exclusions      = new Exclusion[numExclusions];
  exclusion_atoms = new int[numExclusions];

  if ( (exclusions == NULL) || (exclusion_atoms == NULL) )
  {
    NAMD_die("memory allocation failed in Molecule::read_exclusions");
  }

  /*  First, read in the excluded atoms list      */
  for (num_read=0; num_read<numExclusions; num_read++)
  {
    /*  Read the atom number from the file. Subtract 1 to   */
    /*  convert the index from the 1 to NumAtoms used in the*/
    /*  file to the  0 to NumAtoms-1 that we need    */
    exclusion_atoms[num_read]=NAMD_read_int(fd, "IMPROPERS")-1;

    /*  Check for an illegal index        */
    if (exclusion_atoms[num_read] >= numAtoms)
    {
      char err_msg[128];

      sprintf(err_msg, "EXCLUSION INDEX %d GREATER THAN NATOM %d IN EXCLUSION # %d IN PSF FILE", exclusion_atoms[num_read]+1, numAtoms, num_read+1);
      NAMD_die(err_msg);
    }
  }

  /*  Now, go through and read the list of NumAtoms pointers into */
  /*  the array that we just read in        */
  last_index=0;

  for (num_read=0; num_read<numAtoms; num_read++)
  {
    /*  Read in the current index value      */
    current_index=NAMD_read_int(fd, "EXCLUSIONS");

    /*  Check for an illegal pointer      */
    if (current_index>numExclusions)
    {
      char err_msg[128];

      sprintf(err_msg, "EXCLUSION INDEX %d LARGER THAN NUMBER OF EXLCUSIONS %d IN PSF FILE, EXCLUSION #%d\n", 
         current_index+1, numExclusions, num_read);
      NAMD_die(err_msg);
    }

    /*  Check to see if it matches the last index.  If so   */
    /*  than this atom has no exclusions.  If not, then     */
    /*  we have to build some exclusions      */
    if (current_index != last_index)
    {
      /*  This atom has some exlcusions.  Loop from   */
      /*  the last_index to the current index.  This  */
      /*  will include how ever many exclusions this  */
      /*  atom has          */
      for (insert_index=last_index; 
           insert_index<current_index; insert_index++)
      {
        /*  Assign the two atoms involved.      */
        /*  The first one is our position in    */
        /*  the list, the second is based on    */
        /*  the pointer into the index list     */
        int a1 = num_read;
        int a2 = exclusion_atoms[insert_index];
        if ( a1 < a2 ) {
          exclusions[insert_index].atom1 = a1;
          exclusions[insert_index].atom2 = a2;
        } else if ( a2 < a1 ) {
          exclusions[insert_index].atom1 = a2;
          exclusions[insert_index].atom2 = a1;
        } else {
          char err_msg[128];
          sprintf(err_msg, "ATOM %d EXCLUDED FROM ITSELF IN PSF FILE\n", a1+1);
          NAMD_die(err_msg);
        }
	//printf("Explicit exclusion of %i %i\n", a1, a2);
      }

      last_index=current_index;
    }
  }

  /*  Free our temporary list of indexes        */
  delete [] exclusion_atoms;

  return;
}
/*      END OF FUNCTION read_exclusions      */

/************************************************************************/
/*                  */
/*      FUNCTION print_atoms        */
/*                  */
/*  print_atoms prints out the list of atoms stored in this object. */
/*  It is inteded mainly for debugging purposes.      */
/*                  */
/************************************************************************/

void Molecule::print_atoms(Parameters *params)

{
  register int i;
  double sigma;
  double epsilon;
  double sigma14;
  double epsilon14;

  cout << "ATOM LIST\n" 
      << "******************************************\n" 
                  << "NUM  NAME TYPE RES  MASS    CHARGE SIGMA   EPSILON SIGMA14 EPSILON14\n" 
      << endl;

  for (i=0; i<numAtoms; i++)
  {
    params->get_vdw_params(&sigma, &epsilon, &sigma14, &epsilon14, 
        atoms[i].vdw_type);

    cout << i+1 << " " << atomNames[i].atomname  
              << " " << atomNames[i].atomtype << " " 
              << atomNames[i].resname  << " " << atoms[i].mass  
        << " " << atoms[i].charge << " " << sigma 
        << " " << epsilon << " " << sigma14 
        << " " << epsilon14 
        << endl;
  }
}
/*      END OF FUNCTION print_atoms      */

/************************************************************************/
/*                  */
/*      FUNCTION print_bonds        */
/*                  */
/*  print_bonds prints out the list of bonds stored in this object. */
/*  It is inteded mainly for debugging purposes.      */
/*                  */
/************************************************************************/

void Molecule::print_bonds(Parameters *params)

{
  register int i;
  double k;
  double x0;

  cout << "BOND LIST\n" << "********************************\n" 
      << "ATOM1 ATOM2 TYPE1 TYPE2      k        x0" 
      << endl;

  for (i=0; i<numBonds; i++)
  {
    params->get_bond_params(&k, &x0, bonds[i].bond_type);

    cout << bonds[i].atom1+1 << " " 
       << bonds[i].atom2+1 << " "   
       << atomNames[bonds[i].atom1].atomtype << " "  
       << atomNames[bonds[i].atom2].atomtype << " " << k 
       << " " << x0 << endl;
  }
}
/*      END OF FUNCTION print_bonds      */

/************************************************************************/
/*                  */
/*      FUNCTION print_exclusions      */
/*                  */
/*  print_exlcusions prints out the list of exlcusions stored in    */
/*  this object.  It is inteded mainly for debugging purposes.    */
/*                  */
/************************************************************************/

void Molecule::print_exclusions()
{
  register int i;

  cout << "EXPLICIT EXCLUSION LIST\n" 
      << "********************************\n" 
            << "ATOM1 ATOM2 " 
      << endl;

  for (i=0; i<numExclusions; i++)
  {
    cout << explicitExcl[i].atom1+1 << "  " 
       << explicitExcl[i].atom2+1 << endl;
  }
}
/*      END OF FUNCTION print_exclusions    */


    /************************************************************************/
    /*                  */
    /*      FUNCTION build_lists_by_atom      */
    /*                  */
    /*  This function builds O(NumAtoms) arrays that store the bonds,   */
    /*  angles, dihedrals, and impropers, that each atom is involved in.    */
    /*  This is a space hog, but VERY fast.  This will certainly have to    */
    /*  change to make things scalable in memory, but for now, speed is the */
    /*  thing!                */
    /*                  */
    /************************************************************************/

    void Molecule::build_lists_by_atom()
       
    {
       register int i;      //  Loop counter
       register int numFixedAtoms = this->numFixedAtoms;  // many tests
       
       bondsWithAtom = new int *[numAtoms];
       bondsByAtom = new int *[numAtoms];
       anglesByAtom = new int *[numAtoms];
       dihedralsByAtom = new int *[numAtoms];
       impropersByAtom = new int *[numAtoms];
       exclusionsByAtom = new int *[numAtoms];

       numBondsWithAtom = new int[numAtoms];
       memset((void*) numBondsWithAtom, 0, numAtoms*sizeof(int));

       int *byAtomSize = new int[numAtoms];

       //  Build the bond lists
       for (i=0; i<numAtoms; i++)
       {
         byAtomSize[i] = 0;
       }
       for (i=0; i<numBonds; i++)
       {
         byAtomSize[bonds[i].atom1]++;
         byAtomSize[bonds[i].atom2]++;
	 // count the number of bonds atom is involved in
         numBondsWithAtom[bonds[i].atom1]++;
         numBondsWithAtom[bonds[i].atom2]++;
       }
       for (i=0; i<numAtoms; i++)
       {
         bondsWithAtom[i] = new int[byAtomSize[i]+1];
         bondsWithAtom[i][byAtomSize[i]] = -1;
         byAtomSize[i] = 0;
       }
       for (i=0; i<numBonds; i++)
       {
         int a1 = bonds[i].atom1;
         int a2 = bonds[i].atom2;
         bondsWithAtom[a1][byAtomSize[a1]++] = i;
         bondsWithAtom[a2][byAtomSize[a2]++] = i;
	 //cerr <<"bondsWithAtom="<<bondsWithAtom[a1][byAtomSize[a1]-1]<<endl;
       }
       
       //  Build the bond lists
       numBondsByAtom = new int[numAtoms];
       memset((void*) numBondsByAtom, 0, numAtoms*sizeof(int));
       for (i=0; i<numAtoms; i++)
       {
         byAtomSize[i] = 0;
       }
       numCalcBonds = 0;
       for (i=0; i<numBonds; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[bonds[i].atom1]
                            && fixedAtomFlags[bonds[i].atom2] ) continue;
         byAtomSize[bonds[i].atom1]++;
         numCalcBonds++;
	 // count the number of bonds atom is involved in
         numBondsByAtom[bonds[i].atom1]++;
         numBondsByAtom[bonds[i].atom2]++;
       }
       for (i=0; i<numAtoms; i++)
       {
         bondsByAtom[i] = new int[byAtomSize[i]+1];
         bondsByAtom[i][byAtomSize[i]] = -1;
         byAtomSize[i] = 0;
       }
       for (i=0; i<numBonds; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[bonds[i].atom1]
                            && fixedAtomFlags[bonds[i].atom2] ) continue;
         int a1 = bonds[i].atom1;
         bondsByAtom[a1][byAtomSize[a1]++] = i;
       }
       

       //  Build the angle lists
       numAnglesByAtom = new int[numAtoms];
       memset((void*) numAnglesByAtom, 0, numAtoms*sizeof(int));
       for (i=0; i<numAtoms; i++)
       {
         byAtomSize[i] = 0;
       }
       numCalcAngles = 0;
       for (i=0; i<numAngles; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[angles[i].atom1]
                            && fixedAtomFlags[angles[i].atom2]
                            && fixedAtomFlags[angles[i].atom3] ) continue;
         byAtomSize[angles[i].atom1]++;
         numCalcAngles++; // total tumber of angles

	 // count the number of angles atom is involved in
         numAnglesByAtom[angles[i].atom1]++;
         numAnglesByAtom[angles[i].atom2]++;
         numAnglesByAtom[angles[i].atom3]++;
       }
       // Create a list of lists
       // (List of angles owned by each atom)
       for (i=0; i<numAtoms; i++)
       {
         anglesByAtom[i] = new int[byAtomSize[i]+1];
         anglesByAtom[i][byAtomSize[i]] = -1;  // Sentinel value at the end
         byAtomSize[i] = 0;
       }
       for (i=0; i<numAngles; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[angles[i].atom1]
                            && fixedAtomFlags[angles[i].atom2]
                            && fixedAtomFlags[angles[i].atom3] ) continue;
	 // atom1 always owns the angle
         int a1 = angles[i].atom1;
         anglesByAtom[a1][byAtomSize[a1]++] = i;
       }
    
       //  Build the improper lists
       numImpropersByAtom = new int[numAtoms];
       memset((void*) numImpropersByAtom, 0, numAtoms*sizeof(int));
       for (i=0; i<numAtoms; i++)
       {
         byAtomSize[i] = 0;
       }
       numCalcImpropers = 0;
       for (i=0; i<numImpropers; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[impropers[i].atom1]
                            && fixedAtomFlags[impropers[i].atom2]
                            && fixedAtomFlags[impropers[i].atom3]
                            && fixedAtomFlags[impropers[i].atom4] ) continue;
         byAtomSize[impropers[i].atom1]++;
         numCalcImpropers++;

	 // count the number of impropers atom is involved in
         numImpropersByAtom[impropers[i].atom1]++;
         numImpropersByAtom[impropers[i].atom2]++;
         numImpropersByAtom[impropers[i].atom3]++;
         numImpropersByAtom[impropers[i].atom4]++;
       }
       for (i=0; i<numAtoms; i++)
       {
         impropersByAtom[i] = new int[byAtomSize[i]+1];
         impropersByAtom[i][byAtomSize[i]] = -1;
         byAtomSize[i] = 0;
       }
       for (i=0; i<numImpropers; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[impropers[i].atom1]
                            && fixedAtomFlags[impropers[i].atom2]
                            && fixedAtomFlags[impropers[i].atom3]
                            && fixedAtomFlags[impropers[i].atom4] ) continue;
         int a1 = impropers[i].atom1;
         impropersByAtom[a1][byAtomSize[a1]++] = i;
       }
       
    
       //  Build the dihedral lists
       numDihedralsByAtom = new int[numAtoms];
       memset((void*) numDihedralsByAtom, 0, numAtoms*sizeof(int));
       for (i=0; i<numAtoms; i++)
       {
         byAtomSize[i] = 0;
       }
       numCalcDihedrals = 0;
       for (i=0; i<numDihedrals; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[dihedrals[i].atom1]
                            && fixedAtomFlags[dihedrals[i].atom2]
                            && fixedAtomFlags[dihedrals[i].atom3]
                            && fixedAtomFlags[dihedrals[i].atom4] ) continue;
         byAtomSize[dihedrals[i].atom1]++;
         numCalcDihedrals++;
	 // count the number of dihedralss atom is involved in
         numDihedralsByAtom[dihedrals[i].atom1]++;
         numDihedralsByAtom[dihedrals[i].atom2]++;
         numDihedralsByAtom[dihedrals[i].atom3]++;
         numDihedralsByAtom[dihedrals[i].atom4]++;
       }
       for (i=0; i<numAtoms; i++)
       {
         dihedralsByAtom[i] = new int[byAtomSize[i]+1];
         dihedralsByAtom[i][byAtomSize[i]] = -1;
         byAtomSize[i] = 0;
       }
       for (i=0; i<numDihedrals; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[dihedrals[i].atom1]
                            && fixedAtomFlags[dihedrals[i].atom2]
                            && fixedAtomFlags[dihedrals[i].atom3]
                            && fixedAtomFlags[dihedrals[i].atom4] ) continue;
         int a1 = dihedrals[i].atom1;
         dihedralsByAtom[a1][byAtomSize[a1]++] = i;
       }
    
    
       //  Build the arrays of exclusions for each atom
       build_exclusions();

       // Copy explicit exclusions to extra array
       if (explicitExcl != NULL)
	 delete [] explicitExcl;

       explicitExcl = new Exclusion[numExclusions];

       for (i=0; i<numExclusions; i++) {
	 explicitExcl[i]=exclusions[i];
       }


       if (exclusions != NULL)
	 delete [] exclusions;

       // 1-4 exclusions which are also fully excluded were eliminated by hash table
       numTotalExclusions = exclusionSet.size();
       exclusions = new Exclusion[numTotalExclusions];
       UniqueSetIter<Exclusion> exclIter(exclusionSet);
       for ( exclIter=exclIter.begin(),i=0; exclIter != exclIter.end(); exclIter++,i++ )
       {
         exclusions[i] = *exclIter;
       }
       // Free exclusionSet storage
       // exclusionSet.clear(1);
       exclusionSet.clear();

    
       for (i=0; i<numAtoms; i++)
       {
         byAtomSize[i] = 0;
       }
       numCalcExclusions = 0;
       for (i=0; i<numTotalExclusions; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[exclusions[i].atom1]
                            && fixedAtomFlags[exclusions[i].atom2] ) continue;
         byAtomSize[exclusions[i].atom1]++;
         numCalcExclusions++;
       }
       for (i=0; i<numAtoms; i++)
       {
         exclusionsByAtom[i] = new int[byAtomSize[i]+1];
         exclusionsByAtom[i][byAtomSize[i]] = -1;
         byAtomSize[i] = 0;
       }
       for (i=0; i<numTotalExclusions; i++)
       {
         if ( numFixedAtoms && fixedAtomFlags[exclusions[i].atom1]
                            && fixedAtomFlags[exclusions[i].atom2] ) continue;
         int a1 = exclusions[i].atom1;
         exclusionsByAtom[a1][byAtomSize[a1]++] = i;
       }

       //  Allocate an array of int *'s to hold the exclusions for
       //  each atom
       all_exclusions = new int *[numAtoms];

       for (i=0; i<numAtoms; i++)
       {
         byAtomSize[i] = 0;
       }
       for (i=0; i<numTotalExclusions; i++)
       {
         // first atom should alway have lower number!
         if ( ! exclusions[i].modified )
         {
            if ( numFixedAtoms && fixedAtomFlags[exclusions[i].atom1]
                               && fixedAtomFlags[exclusions[i].atom2] ) continue;
            byAtomSize[exclusions[i].atom1]++;
         }
       }
       for (i=0; i<numAtoms; i++)
       {
         all_exclusions[i] = new int[byAtomSize[i]+1];
         all_exclusions[i][byAtomSize[i]] = -1;
         byAtomSize[i] = 0;
       }
       for (i=0; i<numTotalExclusions; i++)
       {
         if ( ! exclusions[i].modified )
         {
            if ( numFixedAtoms && fixedAtomFlags[exclusions[i].atom1]
                               && fixedAtomFlags[exclusions[i].atom2] ) continue;
            int a1 = exclusions[i].atom1;
            int a2 = exclusions[i].atom2;
            all_exclusions[a1][byAtomSize[a1]++] = a2;
         }
       }

       //  If the exclusion policy is scaled 1-4, then allocate
       //  an array of int *'s to hold the 1-4 interactions
       //  Allocate them all the time and assume they are there! -JCP
       //if (simParams->exclude == EXCLUDE1_4SCALED)
       { 
         onefour_exclusions = new int *[numAtoms];

         for (i=0; i<numAtoms; i++)
         {
            byAtomSize[i] = 0;
         }
         for (i=0; i<numTotalExclusions; i++)
         {
            // first atom should always have lower number!
            if ( exclusions[i].modified )
            {
              if ( numFixedAtoms && fixedAtomFlags[exclusions[i].atom1]
                                 && fixedAtomFlags[exclusions[i].atom2] ) continue;
              byAtomSize[exclusions[i].atom1]++;
            }
         }
         for (i=0; i<numAtoms; i++)
         {
            onefour_exclusions[i] = new int[byAtomSize[i]+1];
            onefour_exclusions[i][byAtomSize[i]] = -1;
            byAtomSize[i] = 0;
         }
         for (i=0; i<numTotalExclusions; i++)
         {
            if ( exclusions[i].modified )
            {
              if ( numFixedAtoms && fixedAtomFlags[exclusions[i].atom1]
                                 && fixedAtomFlags[exclusions[i].atom2] ) continue;
              int a1 = exclusions[i].atom1;
              int a2 = exclusions[i].atom2;
              onefour_exclusions[a1][byAtomSize[a1]++] = a2;
            }
         }
       }

       delete [] byAtomSize;

    }
    /*    END OF FUNCTION build_lists_by_atom    */



/****************************************************************/
/*                */
/*      FUNCTION build_exclusions    */
/*                */
/*  This function builds a list of all the exlcusions       */
/*  atoms.  These lists include explicit exclusions as well as  */
/*  exclusions that are calculated based on the bonded structure*/
/*  and the exclusion flag.  For each pair of atoms that are    */
/*  excluded, the larger of the 2 atom indexes is stored in the */
/*  array of the smaller index.  All the arrays are not sorted. */
/*  Then to determine if two atoms have an exclusion, a linear  */
/*  search is done on the array of the atom with the smaller    */
/*  index for the larger index.          */
/*  If the exclusion policy is set to scaled1-4, there are  */
/*  actually two lists built.  One contains the pairs of atoms  */
/*  that are to be exlcuded (i.e., explicit exclusions, 1-2,    */
/*  and 1-3 interactions) and the other contains just the 1-4   */
/*  interactions, since they will need to be determined   */
/*  independantly of the other exclusions.      */
/*                */
/****************************************************************/

void Molecule::build_exclusions()
{
  for (int i=0; i<numExclusions; i++)
    exclusionSet.add(exclusions[i]);
        switch (simParams->exclude)
        {
         case NONE:
           break;
         case EXCLUDE1_2:
           build12excl();
           break;
          case EXCLUDE1_3:
            build12excl();
            build13excl();
            break;
          case EXCLUDE1_4:
            build12excl();
            build13excl();
            build14excl(0);
            break;
          case EXCLUDE1_4SCALED:
            build12excl();
            build13excl();
            build14excl(1);
            break;
        }
}

void Molecule::build12excl(void)
  
{
  int *current_val;  //  Current value to check
  register int i;    //  Loop counter to loop through all atoms
  
  //  Loop through all the atoms marking the bonded interactions for each one
  for (i=0; i<numAtoms; i++)
    {
      current_val = bondsWithAtom[i];
      
      //  Loop through all the bonds for this atom
      while (*current_val != -1)
      {
	if (bonds[*current_val].atom1 == i)
	  {
	    if (i<bonds[*current_val].atom2)
	      {
		exclusionSet.add(Exclusion(i,bonds[*current_val].atom2));
	      }
	  }
	else
	  {
	    if (i<bonds[*current_val].atom1)
	      {
		exclusionSet.add(Exclusion(i,bonds[*current_val].atom1));
	      }
	  }

	++current_val;
      }
    }
}

void Molecule::build13excl()
{
  int *bond1, *bond2;  //  The two bonds being checked
  int middle_atom;  //  Common third atom
  register int i;    //  Loop counter to loop through all atoms
  
  //  Loop through all the atoms looking at the bonded connections
  //  for each one
  for (i=0; i<numAtoms; i++)
    {
      bond1 = bondsWithAtom[i];
      
      //  Loop through all the bonds directly connect to atom i
      while (*bond1 != -1)
	{
	  if (bonds[*bond1].atom1 == i)
	    {
	      middle_atom=bonds[*bond1].atom2;
	    }
	  else
	    {
	      middle_atom=bonds[*bond1].atom1;
	    }
	  
	  bond2 = bondsWithAtom[middle_atom];
	  
	  //  Now loop through all the bonds connect to the
	  //  middle atom
	  while (*bond2 != -1)
	    {
	      if (bonds[*bond2].atom1 == middle_atom)
		{
		  if (i < bonds[*bond2].atom2)
		    {
		      exclusionSet.add(Exclusion(i,bonds[*bond2].atom2));
		    }
		}
	      else
		{
		  if (i < bonds[*bond2].atom1)
		    {
		      exclusionSet.add(Exclusion(i,bonds[*bond2].atom1));
		    }
		}
	      
	      ++bond2;
	    }
	  
	  ++bond1;
	}
    }
}
/*      END OF FUNCTION build13excl      */

/************************************************************************/
/*                  */
/*        FUNCTION build14excl      */
/*                  */
/*   INPUTS:                */
/*  lists - Array of IntList objects to put exclusions into    */
/*                  */
/*  This function calculates all the 1-4 exclusions (that is,  */
/*   atoms that are connected via a sequence of three linear bonds) and */
/*   places these interactions into the array of IntList object passed  */
/*   in.                */
/*                  */
/************************************************************************/


void Molecule::build14excl(int modified)
  
{
  int *bond1, *bond2, *bond3;  //  The two bonds being checked
  int mid1, mid2;    //  Middle atoms
  register int i;      //  Counter to loop through all atoms
  
  //  Loop through all the atoms
  for (i=0; i<numAtoms; i++)
    {  
      // Get all the bonds connect directly to atom i
      bond1 = bondsWithAtom[i];
      
      while (*bond1 != -1)
	{
	  if (bonds[*bond1].atom1 == i)
	    {
	      mid1=bonds[*bond1].atom2;
	    }
	  else
	    {
	      mid1=bonds[*bond1].atom1;
	    }
	  
	  bond2 = bondsWithAtom[mid1];
	  
	  //  Loop through all the bonds connected to atom mid1
	  while (*bond2 != -1)
	    {
	      if (bonds[*bond2].atom1 == mid1)
		{
		  mid2 = bonds[*bond2].atom2;
		}
	      else
		{
		  mid2 = bonds[*bond2].atom1;
		}
	      
	      //  Make sure that we don't double back to where
	      //  we started from.  This causes strange behavior.
	      //  Trust me, I've been there . . .
	      if (mid2 == i)
		{
		  ++bond2;
		  continue;
		}
	      
	      bond3=bondsWithAtom[mid2];
	      
	      //  Loop through all the bonds connected to mid2
	      while (*bond3 != -1)
		{
		  if (bonds[*bond3].atom1 == mid2)
		    {
		      //  Make sure that we don't double back to where
		      //  we started from.  This causes strange behavior.
		      //  Trust me, I've been there . . .
		      //  I added this!!!  Why wasn't it there before?  -JCP
		      if (bonds[*bond3].atom2 != mid1)
			if (i<bonds[*bond3].atom2)
			  {
			    exclusionSet.add(Exclusion(i,bonds[*bond3].atom2,modified));
			  }
		    }
		  else
		    {
		      //  Make sure that we don't double back to where
		      //  we started from.  This causes strange behavior.
		      //  Trust me, I've been there . . .
		      //  I added this!!!  Why wasn't it there before?  -JCP
		      if (bonds[*bond3].atom1 != mid1)
			if (i<bonds[*bond3].atom1)
			  {
			    exclusionSet.add(Exclusion(i,bonds[*bond3].atom1,modified));
			  }
		    }
		  
		  ++bond3;
		}
	      
	      ++bond2;
	    }
	  
	  ++bond1;
	}
    }
}
    /*      END OF FUNCTION build14excl      */


    /************************************************************************/
    /*                                                                      */
    /*        FUNCTION stripHGroupExcl                                      */
    /*                                                                      */
    /*  This function removes all exclusions which are entirely             */
    /*  within a single hydrogen group.  This assumes that they all         */
    /*  exist, which should be true for exclusion policy 1-3 or higher.     */
    /*                                                                      */
    /************************************************************************/

  void Molecule::stripHGroupExcl(void)
  {

    HydrogenGroup::iterator h_i, h_e, h_j;
    h_i = hydrogenGroup.begin();  h_e = hydrogenGroup.end();
    for( ; h_i != h_e; ++h_i ) {
      for ( h_j = h_i + 1; h_j != h_e && ! h_j->isGP; ++h_j ) {
	if ( h_i->atomID < h_j->atomID )
	  exclusionSet.del(Exclusion(h_i->atomID,h_j->atomID));
	else
	  exclusionSet.del(Exclusion(h_j->atomID,h_i->atomID));
      }
    }

  }
    /*      END OF FUNCTION stripHGroupExcl      */


int Molecule::is_hydrogen(int anum) {
  return ((atoms[anum].status & HydrogenAtom) != 0);
}

int Molecule::is_oxygen(int anum) {
  return ((atoms[anum].status & OxygenAtom) != 0);
}

int Molecule::is_hydrogenGroupParent(int anum) {
  return (hydrogenGroup[atoms[anum].hydrogenList].isGP);
}

int Molecule::is_water(int anum) {
  return (hydrogenGroup[atoms[anum].hydrogenList].sortVal == 2);
}

int Molecule::get_groupSize(int anum) {
  return (hydrogenGroup[atoms[anum].hydrogenList].atomsInGroup);
}

void Molecule::get_child_atoms(int anum, int* numChildren, int** cID) {
  *numChildren = hydrogenGroup[atoms[anum].hydrogenList].atomsInGroup-1;
  *cID = hydrogenGroup[atoms[anum].hydrogenList].childID;
}


// go through the molecular structure, analyze the status of each atom,
// and save the data in the Atom structures stored for each atom.  This
// could be built up incrementally while the molecule is being read in,
// but doing it all in one shot allows us to just send the basic info
// over the network and have each node calculate the rest of the data on
// it's own.
void Molecule::build_atom_status(void) {
  register int i;
  int a1, a2;
  
  // initialize information for each atom (note that the status has
  // already been initialized during the read/receive phase)
  HydrogenGroupID *hg;
  hg = new HydrogenGroupID[numAtoms];
  for (i=0; i < numAtoms; i++) {
    hg[i].atomID = i;  // currently unsorted
    hg[i].atomsInGroup = 1;  // currently only 1 in group
    hg[i].isGP = 1;  // assume it is a group parent
    hg[i].GPID = i;  // assume it is a group parent
    hg[i].sortVal = 0;  // for group sorting
    int j=0;
    while(j<4) hg[i].childID[j++]=-1;
  }
  
  // find which atom each hydrogen is bound to
  // also determine number of atoms in each group
  for (i=0; i < numBonds; i++) {
    a1 = bonds[i].atom1;
    a2 = bonds[i].atom2;
    if (is_hydrogen(a1))
      {
	// check for hydrogen gas...  For H2, explicitly define the group parent.
	// I have been informed that H3 is not a concern.  This is good since
	// this code will fail for H3.
	if (is_hydrogen(a2)) {
	  hg[a1].isGP = 1;
	  cout <<   "Found H-H bond - are you sure?" << endl;
	} else {
	  hg[a2].childID[hg[a2].atomsInGroup-1] = a1; // add H-atom index to childID list
	  hg[a2].atomsInGroup++;
	  hg[a1].atomsInGroup = 0;
	  hg[a1].GPID = a2;
	  hg[a1].isGP = 0;
	  // check for waters (put them in their own groups: OH or OHH)
	  if (is_oxygen(a2))  hg[a2].sortVal++;
	}
      }
    if (is_hydrogen(a2))
      {
	hg[a1].childID[hg[a1].atomsInGroup-1] = a2; // add H-atom index to childID list
	hg[a1].atomsInGroup++;
	hg[a2].atomsInGroup = 0;
	hg[a2].GPID = a1;
	hg[a2].isGP = 0;
	// check for waters (put them in their own groups: OH or OHH)
	if (is_oxygen(a1))  hg[a1].sortVal++;
      }
  }
  
  // sort the hydrogenGroup list and count number of groups
  numHydrogenGroups = 0;
  for(i=0; i<numAtoms; i++)
  {
    // make H follow their group parents.
    if (!hg[i].isGP)  hg[i].sortVal = hg[hg[i].GPID].sortVal;
    else ++numHydrogenGroups;
    // add to list to sort
    hydrogenGroup.push_back(hg[i]);
  }
  sort(hydrogenGroup.begin(), hydrogenGroup.end());

  // finally, add the indexing from atoms[] to hydrogenGroup[]
  waterIndex = numAtoms;
  for(i=0; i<numAtoms; i++) {
    atoms[hydrogenGroup[i].atomID].hydrogenList = i;
    // identify where waters start
    if ((hydrogenGroup[i].sortVal == 2) && (i < numAtoms))
      waterIndex = i;
  }

#if 0
  // debugging code for showing sorted atoms
  for(i=0; i<numAtoms; i++)
    cout << i << " atomID=" << hydrogenGroup[i].atomID
   << " isGP=" << hydrogenGroup[i].isGP
   << " parent=" << hydrogenGroup[i].GPID
   << " #" << hydrogenGroup[i].atomsInGroup
   << " sortVal=" << hydrogenGroup[i].sortVal
   << "\n" << endl;
#endif

  delete [] hg;
}
/* END FUNCTION build_atom_status */

void Molecule::build_donor_list(Parameters* hbparams) {
  donorsByAtom = new int[numAtoms];
  memset((void*) donorsByAtom, -1, numAtoms*sizeof(int));

  int count=0;

  for (int j=0; j<numHDonors; j++) {
      int atomnum = get_donor(j);
      if (atomnum>=numAtoms) { cerr<<"build_donor_list(): atomnum too high!!"<< endl; }
      if(get_num_child_atoms(atomnum)) {
	  donors[j].type=hbparams->assign_donor_type_index(get_atomtype(atomnum));
	  donorsByAtom[atomnum]=j;
	  count++;
      }
  }
  if (count<numHDonors) NAMD_die("found donor without hydrogen!!");
}

void Molecule::build_acceptor_list(Parameters* hbparams) {
  acceptorsByAtom = new int[numAtoms];
  memset((void*) acceptorsByAtom, -1, numAtoms*sizeof(int));

  int count=0;

  for (int j=0; j<numHAcceptors; j++) {
      int atomnum = get_acceptor(j);
      //cerr << j << ": " << atomnum << endl;
      if (atomnum>=numAtoms) { cerr<<"build_acceptor_list(): atomnum too high!!"<< endl; }
      if(get_num_bonds_for_atom(atomnum)) {
	  //cerr << "atomtype: "<<get_atomtype(atomnum) <<endl;
	  acceptors[j].type=hbparams->assign_accep_type_index(get_atomtype(atomnum));
	  acceptorsByAtom[atomnum]=j;
	  count++;
      }
  }
  if (count<numHAcceptors) NAMD_die("found acceptor without antecedent!!");
}

//  Lookup atom id from segment, residue, and index in residue
int* Molecule::get_atoms_from_segid_and_residue(const char *segid, int resid, int *natoms) const 
{
  int *tmp = new int[numAtoms];
  *natoms=0;
  for (int i=0; i<numAtoms; i++) {
    if (!strcmp(atomNames[i].segname, segid) && atomNames[i].resid==resid) {
      tmp[(*natoms)++] = i;
    }
  } 
  int *index = new int[*natoms];
  for(int j=0; j<(*natoms); j++) {
    index[j] = tmp[j];
  }
  delete [] tmp;
  return index;
}

// Set the fixedAtomsFlags
void Molecule::set_fixed_atoms(int *fixedatoms, int numfixed) {
  numFixedAtoms = numfixed;
  fixedAtomFlags = new int[numAtoms];

  // Initialize
  for (int i=0; i<numAtoms; i++) {
    fixedAtomFlags[i] = 0;
  }
  // Set fixed atoms
  for (int i=0; i<numFixedAtoms; i++) {
    fixedAtomFlags[fixedatoms[i]] = 1;
  }
}


/************************************************************************/
/*                  */
/*      FUNCTION Molecule        */
/*                  */
/*  This is the constructor for reading AMBER topology data    */
/*                  */
/************************************************************************/

Molecule::Molecule(SimParameters *simParams, Parameters *param, Ambertoppar *amber_data)
{
  if ( sizeof(int32) != 4 ) { NAMD_die("sizeof(int32) != 4"); }
  this->simParams = simParams;
  this->params = param;
  /*  Initialize array pointers to NULL  */
  atoms=NULL;
  atomNames=NULL;
  bonds=NULL;
  angles=NULL;
  dihedrals=NULL;
  impropers=NULL;
  donors=NULL;
  acceptors=NULL;
  exclusions=NULL;
  //tmpArena=NULL;
  bondsWithAtom=NULL;
  bondsByAtom=NULL;
  anglesByAtom=NULL;
  dihedralsByAtom=NULL;
  impropersByAtom=NULL;
  exclusionsByAtom=NULL;
  all_exclusions=NULL;
  fixedAtomFlags=NULL;
  rigidBondLengths=NULL;
   
  nameArena = new ObjectArena<char>;
  // nameArena->setAlignment(8);
  //arena = new ObjectArena<int32>;
  // arena->setAlignment(32);
  //exclArena = new ObjectArena<char>;
  // exclArena->setAlignment(32);

  /*  Initialize counts to 0 */
  numAtoms=0;
  numBonds=0;
  numAngles=0;
  numDihedrals=0;
  numImpropers=0;
  numDonors=0;
  numAcceptors=0;
  numExclusions=0;
  numConstraints=0;
  numFixedAtoms=0;
  numRigidBonds=0;
  numFixedRigidBonds=0;
  numMultipleDihedrals=0;
  numMultipleImpropers=0;
  numCalcBonds=0;
  numCalcAngles=0;
  numCalcDihedrals=0;
  numCalcImpropers=0;
  numCalcExclusions=0;

  // Read in parm structure
  read_parm(amber_data);

  // build_lists_by_atom(); // Must now be invoked manually.
}
/*      END OF FUNCTION Molecule      */


/************************************************************************/
/*                  */
/*      FUNCTION read_parm   */
/*                  */
/*   INPUTS:                */
/*  amber_data - AMBER data structure    */
/*                  */
/*  This function copys AMBER topology data to the corresponding data  */
/*   structures      */
/*                  */
/************************************************************************/

void Molecule::read_parm(Ambertoppar *amber_data)
{
  int i,j,ntheth,nphih,current_index,a1,a2,
      max,min,index,found;

  if (!amber_data->data_read)
    NAMD_die("No data read from parm file yet!");

  // Copy atom informations
  numAtoms = amber_data->Natom;
  atoms = new Atom[numAtoms];
  atomNames = new AtomNameInfo[numAtoms];
  if (atoms == NULL || atomNames == NULL )
    NAMD_die("memory allocation failed when reading atom information");
  for (i=0; i<numAtoms; ++i)
  { atomNames[i].resname = nameArena->getNewArray(5);
    atomNames[i].atomname = nameArena->getNewArray(5);
    atomNames[i].atomtype = nameArena->getNewArray(5);
    if (atomNames[i].resname == NULL || atomNames[i].atomname == NULL || atomNames[i].atomtype == NULL)
      NAMD_die("memory allocation failed when reading atom information");
    for (j=0; j<4; ++j)
    { atomNames[i].resname[j] = amber_data->ResNames[amber_data->AtomRes[i]*4+j];
      atomNames[i].atomname[j] = amber_data->AtomNames[i*4+j];
      atomNames[i].atomtype[j] = amber_data->AtomSym[i*4+j];
    }
    atomNames[i].resname[4] = atomNames[i].atomname[4] = atomNames[i].atomtype[4] = '\0';
    atoms[i].mass = amber_data->Masses[i];
    // Divide by 18.2223 to convert to charge in units of the electron charge
    atoms[i].charge = amber_data->Charges[i] / 18.2223;
    atoms[i].vdw_type = amber_data->Iac[i] - 1;
    /*  Determine the type of the atom (H or O) */
    atoms[i].status = UnknownAtom; // the default
    if (atoms[i].mass <=3.5) {
      atoms[i].status |= HydrogenAtom;
    } else if ((atomNames[i].atomname[0] == 'O') && 
         (atoms[i].mass >= 14.0) && 
         (atoms[i].mass <= 18.0)) {
      atoms[i].status |= OxygenAtom;
    }
  }

// Note: In AMBER, the atom numbers in bond, angle and dihedral arrays are in fact
// (3*(atnum-1)). So we divide them by 3 to get the real indices of atom array. Also
// note that NAMD indexes arrays from 0 to NumAtoms-1.

  // Copy bond information
  // Fake bonds (bonds with 0 force constant) are ignored
  double k, x0;
  numBonds = 0;
  if (amber_data->Nbonh + amber_data->Nbona > 0)
  { bonds = new Bond[amber_data->Nbonh + amber_data->Nbona];
    if (bonds == NULL || amber_data->Nbona < 0)
      NAMD_die("memory allocation failed when reading bond information");
    // Bonds WITH hydrogen
    for (i=0; i<amber_data->Nbonh; ++i)
    { bonds[numBonds].atom1 = amber_data->BondHAt1[i] / 3;
      bonds[numBonds].atom2 = amber_data->BondHAt2[i] / 3;
      bonds[numBonds].bond_type = amber_data->BondHNum[i] - 1;
      if (bonds[numBonds].atom1>=numAtoms || bonds[numBonds].atom2>=numAtoms ||
          bonds[numBonds].bond_type>=amber_data->Numbnd)
      { char err_msg[128];
        sprintf(err_msg, "BOND (WITH H) # %d OVERFLOW IN PARM FILE", i+1);
        NAMD_die(err_msg);
      }
      params->get_bond_params(&k,&x0,bonds[numBonds].bond_type);
      if ( k != 0. ) ++numBonds;  // real bond
    }
    // Bonds WITHOUT hydrogen
    for (i=amber_data->Nbonh; i<amber_data->Nbonh+amber_data->Nbona; ++i)
    { bonds[numBonds].atom1 = amber_data->BondAt1[i-amber_data->Nbonh] / 3;
      bonds[numBonds].atom2 = amber_data->BondAt2[i-amber_data->Nbonh] / 3;
      bonds[numBonds].bond_type = amber_data->BondNum[i-amber_data->Nbonh] - 1;
      if (bonds[i].atom1>=numAtoms || bonds[i].atom2>=numAtoms ||
          bonds[i].bond_type>=amber_data->Numbnd)
      { char err_msg[128];
        sprintf(err_msg, "BOND (WITHOUT H) # %d OVERFLOW IN PARM FILE", i+1-amber_data->Nbonh);
        NAMD_die(err_msg);
      }
      params->get_bond_params(&k,&x0,bonds[numBonds].bond_type);
      if ( k != 0. ) ++numBonds;  // real bond
    }
  }
  /*  Tell user about our subterfuge  */
  if ( numBonds !=  amber_data->Nbonh + amber_data->Nbona) {
    cout << "Warning: Ignored " << amber_data->Nbonh + amber_data->Nbona - numBonds <<
            " bonds with zero force constants.\n" << endl;
    cout << "Warning: " <<
	"Will get H-H distance in rigid H2O from H-O-H angle.\n" << endl;
  }

  // Copy angle information
  numAngles = amber_data->Ntheth + amber_data->Ntheta;
  if (numAngles > 0)
  { ntheth = amber_data->Ntheth;
    angles = new Angle[numAngles];
    if (angles == NULL || numAngles < ntheth)
      NAMD_die("memory allocation failed when reading angle information");
    // Angles WITH hydrogen
    for (i=0; i<ntheth; ++i)
    { angles[i].atom1 = amber_data->AngleHAt1[i] / 3;
      angles[i].atom2 = amber_data->AngleHAt2[i] / 3;
      angles[i].atom3 = amber_data->AngleHAt3[i] / 3;
      angles[i].angle_type = amber_data->AngleHNum[i] - 1;
      if (angles[i].atom1>=numAtoms || angles[i].atom2>=numAtoms ||
          angles[i].atom3>=numAtoms || angles[i].angle_type>=amber_data->Numang)
      { char err_msg[128];
        sprintf(err_msg, "ANGLE (WITH H) # %d OVERFLOW IN PARM FILE", i+1);
        NAMD_die(err_msg);
      }
    }
    // Angles WITHOUT hydrogen
    for (i=ntheth; i<numAngles; ++i)
    { angles[i].atom1 = amber_data->AngleAt1[i-ntheth] / 3;
      angles[i].atom2 = amber_data->AngleAt2[i-ntheth] / 3;
      angles[i].atom3 = amber_data->AngleAt3[i-ntheth] / 3;
      angles[i].angle_type = amber_data->AngleNum[i-ntheth] - 1;
      if (angles[i].atom1>=numAtoms || angles[i].atom2>=numAtoms ||
          angles[i].atom3>=numAtoms || angles[i].angle_type>=amber_data->Numang)
      { char err_msg[128];
        sprintf(err_msg, "ANGLE (WITHOUT H) # %d OVERFLOW IN PARM FILE", i+1-ntheth);
        NAMD_die(err_msg);
      }
    }
  }

  numExclusions =  0;
  // If readExclusions is TRUE, then we copy exclusions from parm
  // file; otherwise we skip the exclusions here and generate
  // them later in build_exclusions()
  if (simParams->readExclusions)
  { // Copy exclusion information
    // In Amber data structure, Iblo[] is the number of exclusions
    // for each atom; ExclAt[] is the atom index for the excluded atoms.
    exclusions = new Exclusion[amber_data->Nnb];
    if (exclusions == NULL &&  amber_data->Nnb > 0)
      NAMD_die("memory allocation failed when reading exclusion information");
    current_index = 0;
    for (i=0; i<numAtoms; ++i)
      for (j=0; j<amber_data->Iblo[i]; ++j)
      { if (current_index >= amber_data->Nnb)
        { char err_msg[128];
          sprintf(err_msg, "EXCLUSION INDEX EXCEEDS NUMBER OF EXLCUSIONS %d IN AMBER FILE, AT ATOM #%d\n",
            amber_data->Nnb, i+1);
          NAMD_die(err_msg);
        }
        // There's some 0 in the ExclAt[] list, which is strange
        // and redundant. In this case, I simply ignore such entries.
        if (amber_data->ExclAt[current_index] != 0)
        { // Subtract 1 to convert the index from the 1 to NumAtoms
          // used in the file to the 0 to NumAtoms-1 that we need
          a2 = amber_data->ExclAt[current_index] - 1;
          if (a2 < i)
          { // I assume the latter index be larger than the former
            // one, so that the same exclusion won't be double-counted;
            // if not, give error
            char err_msg[128];
            sprintf(err_msg, "Atom #%d has exclusion with atom #%d, in reverse order.", i+1, a2+1);
            NAMD_die(err_msg);
          }
          else if (a2 == i)
          { char err_msg[128];
            sprintf(err_msg, "ATOM %d EXCLUDED FROM ITSELF IN AMBER FILE\n", i+1);
              NAMD_die(err_msg);
          }
          else if (a2 >= numAtoms)
          {  char err_msg[128];
             sprintf(err_msg, "EXCLUSION INDEX %d GREATER THAN NATOM %d IN EXCLUSION # %d IN AMBER FILE",
               a2+1, numAtoms, current_index+1);
             NAMD_die(err_msg);
          }
          exclusions[numExclusions].atom1 = i;
          exclusions[numExclusions].atom2 = a2;
          ++numExclusions;
        }
        ++current_index;
      }
    if (current_index < amber_data->Nnb)
    { char err_msg[128];
      sprintf(err_msg, "Num of exclusions recorded (%d) is smaller than what it's supposed to be (%d)",
        current_index,amber_data->Nnb);
      NAMD_die(err_msg);
    }
  }

  // Copy dihedral information
  numDihedrals = amber_data->Nphih + amber_data->Nphia;
  if (numDihedrals > 0)
  { nphih = amber_data->Nphih;
    dihedrals = new Dihedral[numDihedrals];
    if (dihedrals == NULL || numDihedrals < nphih)
      NAMD_die("memory allocation failed when reading dihedral information");
    // Dihedral WITH hydrogen
    for (i=0; i<nphih; ++i)
    { dihedrals[i].atom1 = amber_data->DihHAt1[i] / 3;
      dihedrals[i].atom2 = amber_data->DihHAt2[i] / 3;
      dihedrals[i].atom3 = amber_data->DihHAt3[i] / 3;
      dihedrals[i].atom4 = amber_data->DihHAt4[i] / 3;
      dihedrals[i].dihedral_type = amber_data->DihHNum[i] - 1;
    }
    // Dihedral WITHOUT hydrogen
    for (i=nphih; i<numDihedrals; ++i)
    { dihedrals[i].atom1 = amber_data->DihAt1[i-nphih] / 3;
      dihedrals[i].atom2 = amber_data->DihAt2[i-nphih] / 3;
      dihedrals[i].atom3 = amber_data->DihAt3[i-nphih] / 3;
      dihedrals[i].atom4 = amber_data->DihAt4[i-nphih] / 3;
      dihedrals[i].dihedral_type = amber_data->DihNum[i-nphih] - 1;
    }
  }
  // In AMBER parm file, dihedrals contain 1-4 exclusion infomation:
  // the 1st and 4th atoms have 1-4 nonbond interation. So we should
  // find them in the exclusion array and change their exclusion to
  // 1-4 type. However, there're two exceptions --
  // 1.If the third atom is negative, it means the end group
  //   interactions are to be ignored;
  // 2.If the fourth atom is negative, it means this is an improper.
  // For the above two cases, the actual atom index is the absolute
  // value of the atom number read; and there's no 1-4 interation
  // for these dihedrals.
  // If readExclusions is not TRUE, then we don't worry about
  // exclusions here.
  for (i=0; i<numDihedrals; ++i)
  { if (dihedrals[i].atom3 < 0 || dihedrals[i].atom4 < 0)
    { dihedrals[i].atom3 = abs(dihedrals[i].atom3);
      dihedrals[i].atom4 = abs(dihedrals[i].atom4);
    }
    else if (simParams->readExclusions)
    { if (dihedrals[i].atom1 < dihedrals[i].atom4)
        a1=dihedrals[i].atom1, a2=dihedrals[i].atom4;
      else
        a1=dihedrals[i].atom4, a2=dihedrals[i].atom1;
      // Since in the exclusion array, atom1 is guaranteed to be
      // ordered, we can do a binary serch to find it first.
      found = 0;
      min=0, max=numExclusions-1;
      while (!found && min<=max)
      { index = (min+max)/2;
        if (exclusions[index].atom1 == a1)
          found = 1;
        else if (exclusions[index].atom1 < a1)
          min = index+1;
        else
          max = index-1;
      }
      if (!found)
        NAMD_die("1-4 interaction in dihedral not found in exclusion list!");
      // After finding atom1, we do a linear serch to find atom2,
      // in both directions.
      for (j=index-1; j>=0 && exclusions[j].atom2!=a2 && exclusions[j].atom1==a1; --j);
      if (j<0 || exclusions[j].atom1!=a1)
        for (j=index; j<numExclusions && exclusions[j].atom2!=a2 && exclusions[j].atom1==a1; ++j);
      if (j<numExclusions && exclusions[j].atom1==a1)
        exclusions[j].modified = 1;  // Change the exclusion type to 1-4
      else
        NAMD_die("1-4 interaction in dihedral not found in exclusion list!");
    }
    if (dihedrals[i].atom1>=numAtoms || dihedrals[i].atom2>=numAtoms ||
        dihedrals[i].atom3>=numAtoms || dihedrals[i].atom4>=numAtoms ||
        dihedrals[i].dihedral_type>=amber_data->Nptra)
    { char err_msg[128];
      sprintf(err_msg, "DIHEDRAL # %d OVERFLOW IN PARM FILE", i+1);
      NAMD_die(err_msg);
    }
  }
  
  //  analyze the data and find the status of each atom
  build_atom_status();

}
/*      END OF FUNCTION read_parm    */
