// LoadDCD.C
// Reads the DCD file and prints the Coords

#include <stdio.h>
//#include <iostream>
#include "Timestep.h"
#include "LoadDCD.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif


// constructor
DCD::DCD() {
  nAtoms = 0;
  nFrames = 0;  
  currFrames = 0;

  namnf = 0;        // number of free atoms
  timeStep = 0.0;
  frameStart = 0;
  frameSkip = 0;  

  fileType = BINARY;  // by default, assume in binary format
  fileptr = NULL;
  Initialized = 0;
  charmmFlags=0;
  reverseEndian=0;

  X = Y = Z = NULL;
  freeatoms = NULL;
}

// close the input file; return success
DCD::~DCD(void) {
  if(opened()) {
    if(file_type() == ASCII)
      fclose(fileptr);
    else
      close_dcd_read(fileptr, namnf, freeatoms);
  }
  if(X) delete [] X;
  if(Y) delete [] Y;
  if(Z) delete [] Z;
}


int DCD::openread(const char* filename)
{
  int errcode, dcdatoms;

  if(opened())
    return TRUE;  // if file is already open, we can continue processing it.

  // initially we assume init failed;
  Initialized = FALSE;
    
  // open the file
  // only try to open as binary 
  fileptr = open_dcd_read(filename);
  if (fileptr == NULL) 
    {
      cerr << "ERROR: (LoadDCD) DCD file not open";
      cerr << "       Filename=" << filename;
      return FALSE;
    }
  fileType = BINARY;
  
  errcode = read_dcdheader(fileptr, &dcdatoms, &nFrames, &frameStart,
			     &frameSkip, &timeStep, &namnf, &freeatoms, 
			     &reverseEndian, &charmmFlags);

  nAtoms = dcdatoms; // only for debug. nAtoms should be spec. by PSF-file
  
  if (errcode < 0 || dcdatoms <= 0) 
    {
      if (errcode == DCD_BADFORMAT)
        cerr << "ERROR: (LoadDCD) Improper format for DCD file";
      else if (errcode == DCD_BADMALLOC)
	cerr << "ERROR: (LoadDCD) DCD file not open";
      else if (dcdatoms <= 0) 
	cerr << "ERROR: (LoadDCD) No atoms found in DCD file";
      else
	cerr << "ERROR: (LoadDCD) Early end to DCD file";
      return FALSE;
    } 
  X = new float[nAtoms];
  Y = new float[nAtoms];
  Z = new float[nAtoms];
  
  return (Initialized = TRUE);
}


int DCD::openwrite(char* filename, int nAtoms, int nFrames, int frameStart,
			     int frameSkip, int timeStep)
{
  int errcode;
  
  this->nAtoms=nAtoms;
  this->nFrames=nFrames;

  if(opened())
    return TRUE;  // if file is already open, we can continue processing it.

  // initially we assume init failed;
  Initialized = FALSE;
    
  // open the file
  // only try to open as binary 
  fileptr = open_dcd_write(filename);
  if (fileptr == NULL) 
    {
      cerr << "ERROR: DCD file not open";
      return FALSE;
    }
  fileType = BINARY;
  
  errcode = write_dcdheader(fileptr, filename, nAtoms, nFrames, frameStart,
			     frameSkip, timeStep);

  if (!errcode) {
      cout << "wrote dcd header\n";
      Initialized = TRUE;
  } 

  return Initialized;
}

void DCD::closewrite() {
    // If the wrong or no number of frames was specified
    // rewind and write correct number of frames into the header.
    if (nFrames != currFrames) {
	fseek(fileptr, 4+sizeof(int), SEEK_SET);
	fwrite((char *) &nAtoms, sizeof(int), 1, fileptr);
    }
    close_dcd_write(fileptr);
}

// Read the next set of coordinates; return list of positions, or NULL if
// error
Timestep* DCD::read(void) {
  int i,j;
  Timestep *pos;
  float *myx = X, *myy = Y, *myz = Z;

  if(!opened() || (nFrames && currFrames == nFrames))
    return NULL;

  // read in next step, if available
  i = read_dcdstep(fileptr, nAtoms, X, Y, Z, namnf,
  	(currFrames == 0), freeatoms, reverseEndian, charmmFlags);
  if(i < 0) {
    if(i == (-1)) {
      cerr << "MDENERGY> End of DCD file reached." << endl;
      return NULL;
    }
    else if(i == DCD_BADFORMAT)
      cerr << "ERROR: Improper format for DCD file";
    else
      cerr << "ERROR: Early end to DCD file";
    return NULL;
  }
    
  // create a new position list to store the data
  pos = new Timestep(nAtoms, (float) timeStep);
  
  // put data in Timestep
  for(i=0, j=0; i < nAtoms; i++) {
    pos->pos[j++] = *(myx++);
    pos->pos[j++] = *(myy++);
    pos->pos[j++] = *(myz++);
    //cerr<<i<<":  "<<j<<":  "<<*(myz)<< "  "<< pos->pos[j-1]<<endl;
  };

  currFrames++;
  return pos;
}

// Read the specified set of coordinates; return list of positions, or NULL if
// error
Timestep* DCD::read(int frame) {
  int ret=0, i, j;
  Timestep *pos;
  float *myx = X, *myy = Y, *myz = Z;

  if(!opened()) {
    cerr << "ERROR: DCD file not initialized!\n";
    return NULL;
  }
  if (nFrames && frame >= nFrames) {
      cerr << "MDENERGY> End of DCD file reached." << endl;
      //cerr << "ERROR: Requested DCD frame >= total number of frames in file\n";
      //cerr << "       frame = " << nFrames << ",   nFrames = " << nFrames << endl;;
    return NULL;
  }

  // Skip over frames to get to requested reading position
  for (int s=currFrames; s<frame-1; s++) {
    ret = skip_one_frame(fileptr, nAtoms, namnf, 
			 (currFrames == 0), reverseEndian, charmmFlags);
    currFrames++;
    if(ret < 0) {
      if(ret == (-1)) {
	cerr << "MDENERGY> End of DCD file reached." << endl;
	return NULL;
      }
      else if(ret == DCD_BADFORMAT)
	cerr << "ERROR: Improper format for DCD file" << endl;
      else
	cerr << "ERROR: Early end to DCD file" << endl;
      return NULL;
    }
  }

  // Read in next step, if available
  ret = read_dcdstep(fileptr, nAtoms, X, Y, Z, namnf,
		   (currFrames == 0), freeatoms, reverseEndian, charmmFlags);
  if(ret < 0) {
    if(ret == (-1)) {
      cerr << "MDFORCE> End of DCD file reached." << endl;
      return NULL;
    }
    else if(ret == DCD_BADFORMAT)
      cerr << "ERROR: Improper format for DCD file" << endl;
    else
      cerr << "ERROR: Early end to DCD file" << endl;
    return NULL;
  }
  
  // create a new position list to store the data
  pos = new Timestep(nAtoms, (float) timeStep);
  // put data in Timestep
  for(i=0, j=0; i < nAtoms; i++) {
    pos->pos[j++] = *(myx++);
    pos->pos[j++] = *(myy++);
    pos->pos[j++] = *(myz++);
  };
  
  currFrames++;
  return pos;
}


// Write the next set of coordinates; return total number of frames written, 
// or NULL if error
int DCD::write(Timestep& pos) {
  int i,j;
  float *myx, *myy , *myz;

  if(!opened() || (nFrames && currFrames == nFrames))
    return 0;

  for(i=0, j=0; i < nAtoms; i++) {
      *(myx++)=pos.pos[j++];
      *(myy++)=pos.pos[j++];
      *(myz++)=pos.pos[j++];
  };

  // read in next step, if available
  i = write_dcdstep(fileptr, nAtoms, myx, myy, myz);
  if(i < 0) {
    if(i == (-1))
      return 0;
    else if(i == DCD_BADFORMAT)
      cerr << "ERROR: Improper format for DCD file";
    else
      cerr << "ERROR: Early end to DCD file";
    return 0;
  }
    
  currFrames++;
  return currFrames;
}


// write the next set of coordinates; return total number of frames written, 
// or NULL if error
int DCD::write(FILE * fd, int N, float *X, float *Y, float *Z) {
    int i;
    
    if(!opened() || (nFrames && currFrames == nFrames)) {
	cerr << "ERROR: Max frame number reached!\n"; 
	return 0;
    }
    // read in next step, if available
    i = write_dcdstep(fileptr, nAtoms, X, Y, Z);
    if(i < 0) {
	if(i == (-1))
	    return 0;
	else if(i == DCD_BADFORMAT)
	    cerr << "ERROR: Improper format for DCD file";
	else
	    cerr << "ERROR: Early end to DCD file";
	return 0;
    }
    
  currFrames++;
  return currFrames;
}


// Count the frames in DCD file. Return NULL if error.
int DCD::count_frames() {
  int ret=0;
  int numFrames=0;

  if(!opened()) {
    cerr << "ERROR: DCD file not initialized!\n";
    return -1;
  }

  // Skip over frames to get to requested reading position
  while (ret==0) {
    ret = skip_one_frame(fileptr, nAtoms, namnf, 
			 (numFrames == 0), reverseEndian, charmmFlags);
    numFrames++;
    if(ret < 0) {
      if(ret == (-1)) {
	cerr << "MDFORCE> End of DCD file reached." << endl;
	return -1;
      }
      else if(ret == DCD_BADFORMAT)
	cerr << "ERROR: Improper format for DCD file" << endl;
      else
	cerr << "ERROR: Early end to DCD file" << endl;
      return -1;
    }
  }
  rewind();
  return numFrames;
}

