#include <iomanip.h>
#include "global.h"
#include "Energies.h"
#include "Molecule.h"
#include "Vector.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

// For SlidingWindowAverage:
typedef int (*sliding_window_cb)(void *clientdata, const void *ringdata, int curr_frame);

// Count the number of energies that have to be sent to the output:
int count_output_energies(unsigned eflag) {
  int n=0;
  if(eflag & EBOND)   n++;
  if(eflag & EANGL)   n++;
  if(eflag & EDIHE)   n++;
  if(eflag & EIMPR)   n++;
  if(eflag & EVDW)    n++;
  if(eflag & EELEC)   n++; 
  if(eflag & EKIN)    n+=2; // Ekin and temperature
  if(eflag & ECONF)   n++;
  if((eflag & EALL) || (eflag & EKIN)) n++;
  if(eflag & HBOND)   n++;
  return n;
}

void print_energy_titles(unsigned eflag) {
  cout.precision(6);
  cout << "        frame    time"; 
  if(eflag & EBOND)   cout << "          bond"; 
  if(eflag & EANGL)   cout << "         angle"; 
  if(eflag & EDIHE)   cout << "      dihedral"; 
  if(eflag & EIMPR)   cout << "      improper"; 
  if(eflag & EVDW)    cout << "           vdw"; 
  if(eflag & EELEC)   cout << "          elec"; 
  if(eflag & EKIN)    cout << "       kinetic"; 
  if(eflag & ECONF)   cout << "       conform"; 
  if((eflag & EALL) || (eflag & EKIN))  cout << "         total"; 
  if(eflag & HBOND)   cout << "         hbond"; 
  if(eflag & EKIN)    cout << "       temperature"; 
  if(eflag & FBOND)   cout << "         Fbond"; 
  if(eflag & FANGL)   cout << "         Fangl"; 
  if(eflag & FDIHE)   cout << "         Fdihe"; 
  if(eflag & FIMPR)   cout << "         Fimpr"; 
  if(eflag & FCONF)   cout << "         Fconf"; 
  if(eflag & FVDW)    cout << "          Fvdw";
  if(eflag & FELEC)   cout << "         Felec";
  if(eflag & FNONB)   cout << "         Fnonb"; 
  if(eflag & FTOT)    cout << "          Ftot"; 
  cout << endl;
}


int print_energy_output(void* param, const void* data, int Frame)
{
  Etot_clientdata* p = (Etot_clientdata*) param;
  static int fbeg = p->fbeg;
  double* E = (double*) data;
  unsigned eflag = p->eflag;
  if (!(Frame%10)) print_energy_titles(eflag);

  cout << "ENERGY:";
  cout.setf(ios::fixed);
  cout.width(6);  cout << fbeg+Frame;
  cout.width(8);  cout << p->time+(fbeg+Frame - p->frame0)*p->dt;
  cout.precision(PRECISION);
  if(eflag & EBOND) { cout.width(14); cout << E[BOND]; }
  if(eflag & EANGL) { cout.width(14); cout << E[ANGL]; }
  if(eflag & EDIHE) { cout.width(14); cout << E[DIHE]; }
  if(eflag & EIMPR) { cout.width(14); cout << E[IMPR]; }
  if(eflag & EVDW)  { cout.width(14); cout << E[VDW]; }
  if(eflag & EELEC) { cout.width(14); cout << E[ELEC]; }
  if(eflag & EKIN)  { cout.width(14); cout << E[KIN]; }
  if(eflag & ECONF) { cout.width(14); cout << E[CONF]; }
  if((eflag & EALL) || (eflag & EKIN)) { cout.width(14); cout << E[TOT]; }
  if(eflag & HBOND) { cout.width(14); cout << E[HBON]; }
  if(eflag & EKIN)  { cout.width(14); cout << E[TEMP]; }
  if(eflag & FBOND) { cout.width(14); cout << E[F_BOND]; }
  if(eflag & FANGL) { cout.width(14); cout << E[F_ANGL]; }
  if(eflag & FDIHE) { cout.width(14); cout << E[F_DIHE]; }
  if(eflag & FIMPR) { cout.width(14); cout << E[F_IMPR]; }
  if(eflag & FCONF) { cout.width(14); cout << E[F_CONF]; }
  if(eflag & FVDW)  { cout.width(14); cout << E[F_VDW]; }
  if(eflag & FELEC) { cout.width(14); cout << E[F_ELEC]; }
  if(eflag & FNONB) { cout.width(14); cout << E[F_NONB]; }
  if(eflag & FTOT)  { cout.width(14); cout << E[F_TOT]; }
  cout << endl;
  cout << resetiosflags(ios::floatfield);
  return 1;
}

EnergiesByAtom::EnergiesByAtom(Molecule *molptr, int nAtoms, int sel_n, int *selptr) 
{
  natoms = nAtoms;
  mol  = molptr;
  if (sel_n) {
    atomsel = selptr;
    sel_natoms = sel_n;
  }
  else  sel_natoms = natoms;

  B    = new double[natoms]; // Bond
  A    = new double[natoms]; // Angle
  D    = new double[natoms]; // Dihedral
  I    = new double[natoms]; // Improper
  vdw  = new double[natoms]; // VDW energy
  elec = new double[natoms]; // Elec. energy
  nonb = new double[natoms]; // Hbond energy
  kin  = new double[sel_natoms]; // Kin. energy
  tot  = new double[sel_natoms]; // Tot. energy
  conf = new double[sel_natoms];  // conf. energy
  dof  = new int[sel_natoms];     // for scaling of conf
  value = new float[sel_natoms];  // B-values
  memset((void*) B,    0, natoms*sizeof(double));
  memset((void*) A,    0, natoms*sizeof(double));
  memset((void*) D,    0, natoms*sizeof(double));
  memset((void*) I,    0, natoms*sizeof(double));
  memset((void*) vdw,  0, natoms*sizeof(double));
  memset((void*) elec, 0, natoms*sizeof(double));
  memset((void*) nonb, 0, natoms*sizeof(double));
  memset((void*) kin,  0, sel_natoms*sizeof(double));
  memset((void*) tot,  0, sel_natoms*sizeof(double));
  memset((void*) conf, 0, sel_natoms*sizeof(double));
  memset((void*) dof,  0, sel_natoms*sizeof(int));
  memset((void*) value,0, sel_natoms*sizeof(float));
}

EnergiesByAtom::~EnergiesByAtom()
{
  delete [] B;
  delete [] A;
  delete [] D;
  delete [] I;
  delete [] vdw;
  delete [] elec;
  delete [] nonb;
  delete [] kin;
  delete [] tot;
  delete [] conf;  
  delete [] dof;
  delete [] value;
}


void EnergiesByAtom::compute_dof_by_atom()
{
  /* get list of all conformations for the atom */
  int i;
  for (i=0; i<sel_natoms; i++) {
    int sel_i = atomsel[i];
    int nbonds, nangles, ndihed, nimprop;
    nbonds  = mol->get_num_bonds_for_atom(sel_i);
    nangles = mol->get_num_angles_for_atom(sel_i);
    ndihed  = mol->get_num_dihedrals_for_atom(sel_i);
    nimprop = mol->get_num_impropers_for_atom(sel_i);
    dof[i] = nbonds+nangles+ndihed+nimprop;
    //cout << i << "  index " << sel_i << ":  " << mol->get_resname(sel_i) << ":  " 
    //               << mol->get_atomname(sel_i) << ":  ";
    //cout << nbonds << "  ";
    //cout << nangles << "  ";
    //cout << ndihed << "  ";
    //cout << nimprop << endl;
  }
}

double EnergiesByAtom::compute_kinetic(Vector *vel) 
{
  int j;
  double m_cm = 0;
  double Ekin = 0.0;
  Vector v_cm(0,0,0);

  // center of mass and com-velocities
  for (j=0; j<sel_natoms; j++) {
    v_cm += mol->atommass(j) * vel[j];
    m_cm += mol->atommass(j);
  }
  v_cm /= m_cm;

  // compute kinetic energy
  for (j=0; j<sel_natoms; j++) {
    Vector dv = vel[j] - v_cm;
    kin[j] = 0.5*vel[j]*dv*mol->atommass(j);
    Ekin += kin[j];
  }

  return Ekin;
}

void EnergiesByAtom::compute_conf_and_total() 
{
  for (int i=0; i<sel_natoms; i++) {
    int j = atomsel[i];
    conf[i] = B[j] + A[j] + D[j] + I[j];
    conf[i] /= dof[i];
    //kin[i] -= 1.5*BOLTZMAN*temperature/natoms;
    nonb[i] = vdw[j] + elec[j];
    tot[i] = conf[i] + nonb[i];
  }
}

/*
void select_atoms(int* sel)
{
  atomsel = sel;
}
*/

ForcesByAtom::ForcesByAtom(Molecule *molptr, int nAtoms, int sel_n, int *selptr) 
{
  natoms = nAtoms;
  mol  = molptr;
  if (sel_n) {
    atomsel = selptr;
    sel_natoms = sel_n;
  }
  else  sel_natoms = natoms;

  B     = new Vector[natoms]; // Bond
  A     = new Vector[natoms]; // Angle
  D     = new Vector[natoms]; // Dihedral
  I     = new Vector[natoms]; // Improper
  vdw   = new Vector[natoms]; // VDW force
  elec  = new Vector[natoms]; // Elec. force
  nonb  = new Vector[sel_natoms]; // Hbond force
  tot   = new Vector[sel_natoms]; // Tot. force
  conf  = new Vector[sel_natoms];  // conf. force
  value = new Vector[sel_natoms];  // output force
  abs   = new double[sel_natoms];  // absolute force value
  memset((void*) B,     0, natoms*sizeof(Vector));
  memset((void*) A,     0, natoms*sizeof(Vector));
  memset((void*) D,     0, natoms*sizeof(Vector));
  memset((void*) I,     0, natoms*sizeof(Vector));
  memset((void*) vdw,   0, natoms*sizeof(Vector));
  memset((void*) elec,  0, natoms*sizeof(Vector));
  memset((void*) nonb,  0, sel_natoms*sizeof(Vector));
  memset((void*) tot,   0, sel_natoms*sizeof(Vector));
  memset((void*) conf,  0, sel_natoms*sizeof(Vector));
  memset((void*) value, 0, sel_natoms*sizeof(Vector));
  memset((void*) abs,   0, sel_natoms*sizeof(double));
}


ForcesByAtom::~ForcesByAtom()
{
  delete [] B;
  delete [] A;
  delete [] D;
  delete [] I;
  delete [] vdw;
  delete [] elec;
  delete [] nonb;
  delete [] tot;
  delete [] conf;  
  delete [] value;  
  delete [] abs;
}

void ForcesByAtom::compute_absolute_forces(Vector* f_ptr) {
  for (int i=0; i<sel_natoms; i++) {
    //cerr << i <<" ";
    abs[i] = f_ptr[i].length();
  }
}

void ForcesByAtom::compute_conf_and_total() {
    for (int i=0; i<sel_natoms; i++) {
      int j = atomsel[i];
      conf[i] = B[j] + A[j] + D[j] + I[j];
      nonb[i] = vdw[j] + elec[j];
      tot[i] = conf[i] + nonb[i];
    }
}


