#include <iostream.h> // debug
#include "ComputeNonbonded.h"
#include "Parameters.h"
#include "Molecule.h"
#include "LJTable.h"
#include "DATable.h"
#include "global.h"
#include "Energies.h"
#include "Vector.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

#define VERBOSE
#define HBOND_MIN_ENERGY 0.1

struct atominfo {
  Vector pos;
  Vector vdwforce; 
  Vector elecforce; 
  double vdwenergy; 
  double elecenergy;
  int ind;
};


void ComputeNonbonded::compute( Molecule *mol, const Vector *pos, Vector *vdwf, 
				Vector *elecf, double *evdw, double *eelec, hbond **ehbon,
				double& Evdw, double& Eelec, double& Ehbon, Vector &Fvdw, Vector &Felec,
				const int *selectionByAtom, const int *selection2ByAtom, int have_sel) {

  int xb, yb, zb, xytotb, totb; // dimensions of decomposition

  atominfo **boxatom;           // positions, forces, and indicies for each atom  
  int *numinbox, *maxinbox;     // Number of atoms in each box
  int **nbrlist;                // List of neighbors for each box

  int i, j=0, aindex;

  memset((void *)vdwf,  0, natoms*sizeof(Vector));
  memset((void *)elecf, 0, natoms*sizeof(Vector));
  memset((void *)evdw,  0, natoms*sizeof(double));
  memset((void *)eelec, 0, natoms*sizeof(double));

  //
  // find min/max bounds of molecule's coordinates
  //

  double xmin, xmax, ymin, ymax, zmin, zmax;  // extent of atomic coordinates
  for(i=0; i < natoms; i++) {
    const Vector *loc = pos+i;
    if (selectionByAtom[i] || (have_sel==2 && selection2ByAtom[i])) {
      if(j==0) {
	xmin = xmax = loc->x; ymin = ymax = loc->y; zmin = zmax = loc->z;
      } else {
	if(loc->x < xmin) xmin = loc->x;
	else if(loc->x > xmax) xmax = loc->x;
	if(loc->y < ymin) ymin = loc->y;
	else if(loc->y > ymax) ymax = loc->y;
	if(loc->z < zmin) zmin = loc->z;
	else if(loc->z > zmax) zmax = loc->z;
      }
      j++;
    }
  }

  // from size of molecule, break up space into boxes of dimension pairlistdist
  // Since I'm recreating the pairlist each time, there's no need to make the
  // boxes any bigger than the cutoff length.
  if (selfonly) selshift=0;
  float pairdist = sqrt(cut2);
  xb = (int)((xmax - xmin) / pairdist) + 1 + 2*selshift; // number of boxes in x direction
  yb = (int)((ymax - ymin) / pairdist) + 1 + 2*selshift;
  zb = (int)((zmax - zmin) / pairdist) + 1 + 2*selshift;
  xytotb = yb * xb;   // number of x/y-boxes (in z-slice)
  totb = xytotb * zb; // total number of boxes

  //cout << "Total number of boxes: " << totb << endl;

  boxatom = new atominfo*[totb]; // pointer on array of atominfos for every box
  nbrlist = new int *[totb];     // each box has a list of neighboring boxes
  numinbox = new int[totb];      // actual number of atoms in box
  maxinbox = new int[totb];      // maximum number of atoms in box
  memset((void *)numinbox, 0,totb*sizeof(int));
  memset((void *)maxinbox, 0,totb*sizeof(int));

  //
  // Put all the atoms into their box
  //
  for (i=0; i<natoms; i++) {
    // if (selectionByAtom[i]) {
    const Vector *loc = pos+i; // store position of i-th atom
    // if there's an atomselection than their boxes are centered to allow
    // for the interaction partners in neighbor boxes (selshift=1)
    int axb = (int)(((loc->x - xmin) / pairdist) + selshift); // in which x-box is the atom?
    int ayb = (int)(((loc->y - ymin) / pairdist) + selshift);
    int azb = (int)(((loc->z - zmin) / pairdist) + selshift);

    // Check if atom is in selection or at least in a neighborbox
    if (axb>=0 && axb<xb && ayb>=0 && ayb<yb && azb>=0 && azb<zb) {
      aindex = azb * xytotb + ayb * xb + axb;      // index of the box
      if (numinbox[aindex] == 0) {   // First atom in the box
	maxinbox[aindex] = 10;
	boxatom[aindex] = new atominfo[10]; //start with 10-atom box
	memset((void *)boxatom[aindex], 0, 10*sizeof(atominfo));
      }
      else if (numinbox[aindex] == maxinbox[aindex]) { // Need to resize the box
 	atominfo *tmpbox = new atominfo[2*numinbox[aindex]]; // double size
	memset((void *)tmpbox, 0, 2*numinbox[aindex]*sizeof(atominfo));
	memcpy((void *)tmpbox, (void *)boxatom[aindex], 
	       numinbox[aindex]*sizeof(atominfo));
	delete [] boxatom[aindex];
	boxatom[aindex] = tmpbox;
	maxinbox[aindex] *= 2; // now we have space for more atoms in the box 
      }

      // Now assign the atominfo for each atom
      //    {boxindex}{box related atomindex}
      //      --^--   -------^-------
      boxatom[aindex][numinbox[aindex]].pos = *loc;
      boxatom[aindex][numinbox[aindex]].ind = i;
      numinbox[aindex]++;
    }
  } 
  delete [] maxinbox;
  
  //
  // Create neighbor list for each box
  //
  aindex = 0;
  for (int zi=0; zi<zb; zi++) {
    for (int yi=0; yi<yb; yi++) {
      for (int xi=0; xi<xb; xi++) {
        int nbrs[14];           // Max possible number of neighbors in 3D
        int n=0;                // Number of neighbors found so far
        nbrs[n++] = aindex;     // Always include self
        if (xi < xb-1) nbrs[n++] = aindex + 1;
        if (yi < yb-1) nbrs[n++] = aindex + xb;
        if (zi < zb-1) nbrs[n++] = aindex + xytotb;
        if (xi < (xb-1) && yi < (yb-1)) nbrs[n++] = aindex + xb + 1;
        if (xi < (xb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + 1;
        if (yi < (yb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + xb;
        if (xi < (xb-1) && yi > 0)      nbrs[n++] = aindex - xb + 1;
        if (xi > 0 && zi < (zb-1))     nbrs[n++] = aindex + xytotb - 1;
        if (yi > 0 && zi < (zb-1))     nbrs[n++] = aindex + xytotb - xb;
        if (xi < (xb-1) && yi < (yb-1) && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb + xb + 1;
        if (xi > 0 && yi < (yb-1) && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb + xb - 1; 
        if (xi < (xb-1) && yi > 0 && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb - xb + 1;
        if (xi > 0 && yi > 0 && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb - xb - 1;
        
        nbrlist[aindex] = new int[n+1]; // consists of the boxindices
        memcpy((void *)nbrlist[aindex], (void *)nbrs, n*sizeof(int));
        nbrlist[aindex][n] = -1;  // Sentinel for end of neighbors
        aindex++; // next box
      }
    }
  }

  //
  // Loop over boxes, and compute the interactions between each box and
  // its neighbors.
  //

  Evdw = Eelec = Ehbon = 0.0;
  Fvdw = Felec = 0.0;
#ifdef VERBOSE
  int hbcount = 0; // this is just used in debugging output
#endif

  // loop over boxes
  for (aindex = 0; aindex<totb; aindex++) {
    atominfo *tmpbox = boxatom[aindex];
    int *tmpnbr = nbrlist[aindex];
    //cout << aindex << endl;
    // loop over neighbors of each box
    for (int *nbr = tmpnbr; *nbr != -1; nbr++) {
      atominfo *nbrbox = boxatom[*nbr]; // get the respective atominfos

      // loop over atoms in center box (self atoms)
      for (i=0; i<numinbox[aindex]; i++) {
        register Vector tmpvdwf;                 // temp. vdw force
        register Vector tmpelecf;                // temp. electrostat. force
        register Vector tmppos = tmpbox[i].pos;  // temp. position
	double   tmpevdw  = 0;                   // temp. vdw energy
	double   tmpeelec = 0;                   // temp. elec energy
        int ind1 = tmpbox[i].ind;                // temp. atomindex
        Index vdwtype1 = mol->atomvdwtype(ind1);
        double kq = COULOMB * mol->atomcharge(ind1);
        int startj = 0;
        int num = numinbox[*nbr];

	// loop over atoms in the current neighbor box
	tmpevdw = 0;
        if (aindex == *nbr) startj = i+1;
        for (j=startj; j<num; j++) {
          Vector dr = nbrbox[j].pos - tmppos;  // <--- HERE YOU MUST CHECK IF NEIGHBOR HAS TO BE WRAPPED
          double dist2 = dr.length2();      // distance of the two considered atoms
          if(dist2 > cut2) continue;   
          int ind2 = nbrbox[j].ind;
	  // Are both atoms selected?
	  // If there is only one selection then all entries in selection2ByAtom equal 1.
	  if ((selectionByAtom[ind1] && selection2ByAtom[ind2]) ||
	      (selectionByAtom[ind2] && selection2ByAtom[ind1])) {
	    int bothinfirst;
	    // Are both atoms in the first selection?
	    // !!!!!!!!!!!Selfonly is always zero!!!!!!!!
	    if (selfonly) bothinfirst = selectionByAtom[ind1] && selectionByAtom[ind2];
	    else bothinfirst = 1;  // if -self is not selected they don't have to be in the selection

	    // If both atoms are in the same selection or only one selection is given, go on:
	    if (bothinfirst) {
	      int onefourscaledexcl=0;
	      // check for 1-2, 1-3 and 1-4 exclusions
	      if (exclude==EXCLUDE1_4SCALED && mol->check14excl(ind1, ind2)) { onefourscaledexcl=1; }
	      if (!mol->checkexcl(ind1, ind2) || onefourscaledexcl ) {
		double r = sqrt(dist2);
		double r_1 = 1.0/r; 
		double r_2 = r_1*r_1;
		double r_6 = r_2*r_2*r_2;
		double r_12 = r_6*r_6;
		double switchVal = 1, dSwitchVal = 0;
		double vdwenergy, vdwforce_r;
		if (dist2 > switch2) {
		  double c2 = cut2 - dist2;
		  double c4 = c2*(cut2 + 2*dist2 - 3.0*switch2);
		  switchVal = c2*c4*c1;
		  dSwitchVal = c3*r*(c2*c2-c4);
		}
		
		// get VDW constants
		if (request & EVDW || request & FVDW) {
		  Index vdwtype2 = mol->atomvdwtype(ind2);
		  const LJTableEntry *entry;
		  
		  if (onefourscaledexcl)
		    entry = ljTable->table_val_scaled14(vdwtype1, vdwtype2);
		  else
		    entry = ljTable->table_val(vdwtype1, vdwtype2);
		  double vdwA = entry->A;
		  double vdwB = entry->B;
		  double AmBterm = (vdwA * r_6 - vdwB)*r_6;

		    vdwenergy = switchVal*AmBterm;
		    vdwforce_r = ( switchVal * 6.0 * (vdwA*r_12 + AmBterm) *
				 r_1 - AmBterm*dSwitchVal )*r_1;

		  //if (selectionByAtom[ind1] || selectionByAtom[ind2]) {
		      Evdw += vdwenergy;
		      Fvdw += vdwforce_r * dr;
		      //}
		  tmpvdwf  -= vdwforce_r * dr; 
		  tmpevdw  += vdwenergy * 0.5;
		  nbrbox[j].vdwforce   += vdwforce_r  * dr;
		  nbrbox[j].vdwenergy  += vdwenergy  * 0.5;
		}
		
		// Electrostatics
		if(request & EELEC || request & FELEC) {
		  double kqq = kq * mol->atomcharge(ind2);
		  double efac = 1.0-dist2/cut2;
		  double prefac;
		  if (onefourscaledexcl) {
		    prefac = kqq * r_1 * efac * scale14fac;
		  } else {
		    prefac = kqq * r_1 * efac;
		  }
		  // efac**2 = (1-dist**2/cut**2)**2 
		  // is the XPLOR shifting factor for the elstat potential.
		  double elecforce_r = prefac * r_1 * (r_1 + 3.0*r/cut2);
		  double elecenergy = prefac * efac;

		  // Assuming homogenous electric field along z-axis:
                  // double zground is the z-coordinate where the Potential is zero
		  // It must be defined here or you'd have to pass it into the function
		  // from outside. Or you make it part of the ComputeNonbonded class.
		  // double R_ground = nbrbox[j].pos.z - zground;
		  // double extenergy = kq * mol->atomcharge(ind1) * R_ground;
		  // Vector extfield = 0.0; // Define here or in class ComputeNonbonded
		  // Vector extforce = extfield * mol->atomcharge(ind1);
		  if (selectionByAtom[ind1] || selectionByAtom[ind2]) {
		      Eelec += elecenergy;
		      Felec += elecforce_r * dr; // + extforce;
		  }
		  tmpelecf -= elecforce_r * dr; // + extforce;
		  tmpeelec += elecenergy * 0.5;
		  nbrbox[j].elecforce  += elecforce_r * dr; // + extforce;
		  nbrbox[j].elecenergy += elecenergy * 0.5;
		}
		
		// H-bonds
		if(request & HBOND) {
		  int don, acc, hbond=0;
		  int dd[2], aa[2];
		  if (mol->check_donor(ind1)!=-1 && mol->check_acceptor(ind2)!=-1) {
		    //don=ind1; acc=ind2; 
		    dd[hbond]=ind1; aa[hbond]=ind2; hbond++; 
		  }
		  if (mol->check_donor(ind2)!=-1 && mol->check_acceptor(ind1)!=-1) {
		    //don=ind2; acc=ind1; hbond++;
		    dd[hbond]=ind2; aa[hbond]=ind1; hbond++;
		  }
		  
		  for (int g=0; g<hbond; g++) {
		    don=dd[g]; acc=aa[g];
		    
		    if (hbond && r<DCUT) {
		      int numH, hyd;
		      int *cID;
		      // check_acceptor(acc) returns the index in the acceptor list
		      // get_antecedent(...) gets the antecedent global index for a given acceptor
		      int nant, *ant; 
		      mol->get_antecedents(mol->check_acceptor(acc), &nant, &ant);
		      
		      double tmprHA, rHA = 999999999.9;		  
		      Vector posH, HA, tmpHA;
		      Vector posD  = pos[don];
		      Vector posA  = pos[acc];
		      Vector posAA = pos[*ant];
		      mol->get_child_atoms(don, &numH, &cID);
		      
		      // Select the hydrogen closest to the acceptor
		      for (int w=0;w<numH;w++) {
			tmpHA  = posA-pos[cID[w]];  // hydrogen->acceptor
			tmprHA = tmpHA.length();
			if (tmprHA<rHA) {
			  HA  = tmpHA;  // hydrogen->acceptor
			  rHA = tmprHA;
			  hyd = cID[w];
			  posH = pos[cID[w]];
			}
		      }
		      Vector DH  = posH-posD;  // donor->hydrogen
		      double rDH  = DH.length();
		      double cos_theta_DHA  = DH*HA/(rDH*rHA);
		      double theta_DHA = acos(cos_theta_DHA)/M_PI*180;
		      
		      if (theta_DHA<ACUT) {
			// Compute position of a virtual antecedent, 
                        // if there are more than one antecedents.
			if (nant>1) {
			  int k=0;
			  int nBonds=mol->get_num_bonds_with_atom(acc);
			  int *antelist = new int[nBonds];
			  int *bnum = mol->get_bonds_with_atom(acc);
			  for (int i=0; i<nBonds; i++) {
			    int atom1 = mol->get_bond_atom1(bnum[i]);
			    int atom2 = mol->get_bond_atom2(bnum[i]);
			    if (atom1==acc && atom2!=hyd) antelist[k++]=atom2;
			    else if (atom2==acc && atom1!=hyd) antelist[k++]=atom1;
			  }
			  posAA=posA;
			  for (int m=0; m<k; m++) {
			    //if (mol->get_resid(acc)==268) 
			    //cerr<<"VIRT! "<<mol->get_resid(antelist[m])
			    //    <<"@ "<<mol->get_atomname(antelist[m])
			    //    <<" don="<<mol->get_resid(don) <<endl;
			    Vector AAx = pos[antelist[m]]-posA;
			    posAA += AAx;
			  }
			  delete [] antelist;
			}
			
			Vector AAA = posAA-posA; // acceptor->antecedent
			Vector AD  = posD-posA;  // donor->antecedent
			double rAAA = AAA.length();
			double rAD  = AD.length();
			double cos_theta_HAAA = HA*AAA/(rHA*rAAA);
			//if (mol->get_resid(acc)==268) cerr<<acos(cos_theta_HAAA)/M_PI*180<<endl;
			
			double E;
			const DATableEntry *daEntry;
			//daEntry = daTable->table_val(mol->check_donor(don), mol->check_acceptor(acc));
                        // Get hbond params from type indexes:
			daEntry = daTable->table_val(mol->get_donor_type(mol->check_donor(don)), mol->get_acceptor_type(mol->check_acceptor(acc)));
			E = HB_energy(daEntry->E, daEntry->R, rAD, cos_theta_DHA, cos_theta_HAAA);
			Ehbon += E;
			if (E>HBOND_MIN_ENERGY) {
			  if (verbose) {
			    cerr << hbcount++ << ".)  "
				 << mol->get_resid(don) << ":" << mol->get_atomname(don)
				 <<"---"<< mol->get_resid(hyd) << ":" << mol->get_atomname(hyd)
				 <<"---"<< mol->get_resid(acc) <<":"<< mol->get_atomname(acc);
			    if (nant==1) 
			      cerr <<"---"<< mol->get_resid(*ant) << ":" << mol->get_atomname(*ant);
			    if (nant>1) cerr <<"---"<< mol->get_resid(acc) << ": SEVERAL";
			    cerr << ":  E = " << E << endl;
			    //cerr << daEntry->E <<"  "<< daEntry->R <<"  "<< rDH <<"  "<< rHA<<"  "<< rAAA 
			    //<<"  "<<rAD<<endl; 
			    //cerr << cos_theta_DHA <<"  "<<acos(cos_theta_HAAA)/M_PI*180
			    //   <<"  "<<theta_DHA<<endl;
			    //cerr << endl;
			  }		    
			  // add/insert parameters to the list:
			  if (E>0) add_to_hbond_list(hyd, acc, posH, posA, E);
			}
		      }
		    } // r<DCUT
		  } // g-loop
		} // HBOND
	      } // exclusion check
	    } // bothinfirst (selection)
	  } // selection check
	} // Loop over neighbor atoms

	// Forces and energy in center box
	tmpbox[i].vdwforce += tmpvdwf;
	tmpbox[i].elecforce += tmpelecf; 
	tmpbox[i].vdwenergy += tmpevdw; 
	tmpbox[i].elecenergy += tmpeelec; 
      }     // Loop over self atoms
    }     // Loop over neighbor boxes
  }     // Loop over self boxes
  
  
  
  // 
  // copy forces from atomboxes to the array for export
  //
  for (i = 0; i < totb; i++) {
    for (j=0; j<numinbox[i]; j++) {
      vdwf[boxatom[i][j].ind]  = boxatom[i][j].vdwforce;
      elecf[boxatom[i][j].ind] = boxatom[i][j].elecforce;
      evdw[boxatom[i][j].ind]  = boxatom[i][j].vdwenergy;
      eelec[boxatom[i][j].ind] = boxatom[i][j].elecenergy;
    }
  }

  // free up the storage space allocted for the grid search
  for(i=0; i < totb; i++) {
    if (numinbox[i])  delete [] boxatom[i];
    delete [] nbrlist[i];
  }
  delete [] nbrlist;
  delete [] boxatom;
  delete [] numinbox;


  // adjust size of hbond_list:
  free_hbond_list(get_cur_node());
  *ehbon = hbond_list;
}

//
// The explicit hydrogen-bond term (see X-PLOR manual) has the form
// E = (A/r^6 - B/r^4) * cos^4(theta_A-H-D) * cos^2(theta_AA-A-D)
// involving atoms AA, A, H, D (acceptor antecedent, acceptor, hydrogen, and donor heavy atom).
// while well depth = -B^2/4A and the optimal distance = (2A/B)^(1/6)
//
double ComputeNonbonded::HB_energy(double A, double B, double rAD, double cos_theta_DHA,
				   double cos_theta_HAAA) {
  int i;
  double rAD_AEXP=1, rAD_REXP=1, cos_HAEX=1, cos_AAEX=1;
  double rAD_2 = rAD*rAD;
  double cos_theta_DHA_2 = cos_theta_DHA*cos_theta_DHA;
  for (i=0; i<AEXP; i++) rAD_AEXP *= rAD;
  for (i=0; i<REXP; i++) rAD_REXP *= rAD;
  for (i=0; i<HAEX; i++) cos_HAEX *= cos_theta_DHA;
  for (i=0; i<AAEX; i++) cos_AAEX *= cos_theta_HAAA;
  return (A/rAD_REXP - B/rAD_AEXP)*cos_HAEX*cos_AAEX * rswit(rAD_2)
          *aswit(cos_theta_DHA_2);
}

double ComputeNonbonded::rswit(double r2) {
  const double roff2 = DOFF*DOFF;
  const double ron2 = DON*DON;
  if (r2>roff2) return 0;
  if (r2<ron2)  return 1;
  else {
    double d1 = 1.0/(roff2-ron2);
    d1 = d1*d1*d1;
    double d2 = roff2 - r2;
    double d4 = d2*(roff2 + 2*r2 - 3.0*ron2);
    return d2*d4*d1;
  }
}

double ComputeNonbonded::aswit(double a2) {
  const double aoff2 = cos(M_PI/180*AOFF)*cos(M_PI/180*AOFF);
  const double aon2 = cos(M_PI/180*AON)*cos(M_PI/180*AON);
  if (a2<aoff2) return 0;
  if (a2>aon2)  return 1;
  else {
    double d1 = 1.0/(aoff2-aon2);
    d1 = d1*d1*d1;
    double d2 = aoff2 - a2;
    double d4 = d2*(aoff2 + 2*a2 - 3.0*aon2);
    return d2*d4*d1;
  }
}

/************************************************************************/
/*                                                                      */
/*      FUNCTION add_to_hbond_list                                      */
/*                                                                      */
/*   INPUTS:                                                            */
/*  new_node - node to be added to list                                 */
/*                                                                      */
/*  This function adds a link to the end of the hbond_list list         */
/*                                                                      */
/************************************************************************/

void ComputeNonbonded::add_to_hbond_list(int don, int acc, Vector posDon, Vector posAcc, double E)
{

  //if(prev_node) cerr<<"cur_node->count " <<prev_node->count+1<<endl;

  //  If we are at the end of the list, we must create a new node:
  if (cur_node==NULL)
  {  
    // Allocate a new node:
    struct hbond *new_node = new hbond;
    if (new_node == NULL)
      error_exit("memory allocation failed in ComputeNonbonded::add_hbond\n");
    Vector *dptr = new Vector;
    Vector *aptr = new Vector;
    new_node->posDon = dptr;
    new_node->posAcc = aptr;
    new_node->next   = NULL;
    cur_node         = new_node;
    //new_node->count = numHbonds;

    //cerr << "added hbond " << new_node->count << endl;
    //cerr << "added hbond " << numHbonds << endl;

    //  If the list was empty, then just make the new node the list
    if (hbond_list == NULL) {
      hbond_list     = new_node;
      new_node->prev = NULL;
      //cerr << "begin list " << hbond_list << endl;
    }
    // if it's not empty, add node to end of list:
    else {
      prev_node->next = new_node;
      new_node->prev  = prev_node;
    }
    numHbonds++;
  }

  // Write data into current node:
  //cerr<<"write node"<<cur_node->count<<endl;
  cur_node->don = don;
  cur_node->acc = acc;
  cur_node->posDon[0] = posDon; // strange way to dereference
  cur_node->posAcc[0] = posAcc;
  cur_node->E = E;

  prev_node= cur_node;
  cur_node = cur_node->next; // proceed to next node
}
//      END OF FUNCTION add_to_hbond_list   


/************************************************************************/
/*                                                                      */
/*      FUNCTION free_hbond_list                                        */
/*                                                                      */
/*  This function frees the hbond_list beginning with the node          */
/*  specified in input variable 'first'. So it can be used to adjust    */
/*  the size of the list.                                               */
/*                                                                      */
/************************************************************************/

void ComputeNonbonded::free_hbond_list(struct hbond *last)
{
   struct hbond *ptr, *next;

   ptr=last;          // start deleting with this node
   while (ptr != NULL)
   {
     //cerr << "freeing hbond " << numHbonds-1 << endl;
     next = ptr->next;
     delete ptr->posDon;
     delete ptr->posAcc;
     delete ptr;
     ptr = next;
     numHbonds--;
   }

   if (numHbonds) prev_node->next = NULL;   // terminate list
   else   { hbond_list = NULL;}   // list is empty

   cur_node=hbond_list; // rewind to first node
   prev_node = NULL;    // for the first node exists no previous 
}
/*      END OF FUNCTION free_hbond_list    */


ComputeNonbonded::ComputeNonbonded(const Molecule *mol, 
                                   const Parameters *params,
				   const Parameters *hparams,
                                   const SimParameters *minparams,
				   int *selByAtom, int have_sel) {

  //Molecule *mol = m;
  //Parameters *params = p;
  //Parameters *hparams = hp;
  //SimParameters *simparams = minp;

  if(have_sel==1) selshift = 1;
  else selshift = 0;
  // "request", "selfonly" and "verbose" are set by request_energies() during runtime
  selfonly=0;    // include interaction with environment by default
  verbose=0;

  natoms = mol->numAtoms;
  exclude = minparams->exclude;
  scale14fac = minparams->scale14fac;
  cut2 = minparams->cutoff;
  cut2 *= cut2;
  switch2 = minparams->switchdist;
  switch2 *= switch2;
  pair2 = minparams->pairlistdist;
  pair2 *= pair2;
  c1 = 1.0/(cut2-switch2);
  c1 = c1*c1*c1;
  c3 = 4*c1;
  ljTable = new LJTable(params);
  if (hparams!=NULL) daTable = new DATable(hparams, mol);
  else daTable=NULL;

  numHbonds=0;
  hbond_list=NULL;
  cur_node=NULL;
  prev_node=NULL;
}

ComputeNonbonded::~ComputeNonbonded() {
  delete ljTable;
  if (daTable) delete daTable;

  if (hbond_list != NULL)
    free_hbond_list(hbond_list);
}
