################
# Ionize setup #
################

# Leonardo Trabuco - Tue, 08 Aug 2006

# setupionize:
#
#    Writes a config files for ionize and a pdb file with the charges in the
#    beta field, required for the ionization.
#
# charge2beta:
#
#    Writes a pdb file with the charges in the beta field, required for
#    the ionization. 


package provide setupionize 1.0

namespace eval ::setupionize:: {
  namespace export setupionize
}

proc setupionize { args } { return [eval ::setupionize::setupionize $args] }
proc charge2beta { args } { return [eval ::setupionize::charge2beta $args] }

proc ::setupionize::setupionize_usage { } {

  puts "Setup a config file for runionize."
  puts "Usage: setupionize -psf <psf file> -pdb <pdb file> ?options?"
  puts "Options (defaults on parenthesis):"
  puts "   -conf config file (ionize.conf)"
  puts "   -o pdb file with placed ions (placed_ions.pdb)"
  puts "   -n number of ions to place (1)"
  puts "   -r minimum distance from ions to solute (6.0)"
  puts "   -i minimum distance between placed ions (10.0)"
  puts "   -g grid spacing in angstroms (0.5)"
  puts "   -b additional perimeter to search for ion locations (10.0)"
  puts "   -t ion name (SOD)"
  puts "   -q ion charge (1.0)"
  puts "   -p max_processors (1)"
  return

}

# writes a config file and a pdb with charges in the beta field for ionize
proc ::setupionize::setupionize { args } {

  set nargs [llength $args]
  if { $nargs == 0 || $nargs % 2 } {
    setupionize_usage
    error ""
  }

  foreach {name val} $args {
    switch -- $name {
      -psf { set arg(psf) $val }
      -pdb { set arg(pdb) $val }
      -conf { set arg(conf) $val }
      -o { set arg(o) $val }
      -n { set arg(n) $val }
      -r { set arg(r) $val }
      -i { set arg(i) $val }
      -g { set arg(g) $val }
      -b { set arg(b) $val }
      -t { set arg(t) $val }
      -q { set arg(q) $val }
      -p { set arg(p) $val }
      default { error "unknown argument: $name $val" }
    }
  }

  if { ![info exists arg(psf)] } {
    error "Missing psf file."
  } else {
    set psffile $arg(psf)
  }

  if { ![info exists arg(pdb)] } {
    error "Missing pdb file."
  } else {
    set pdbfile $arg(pdb)
    set pdbcharges "charges-$pdbfile"
  }

  if { ![info exists arg(conf)] } {
    set conffile "ionize.conf"
  } else {
    set conffile $arg(conf)
  }

  if { ![info exists arg(o)] } {
    set pdbions "placed_ions.pdb"
  } else {
    set pdbions $arg(o)
  }

  # write ionize config file
  set conf [open $conffile w]
  puts $conf "PDB_NAME $pdbcharges"
  puts $conf "PDB_OUT $pdbions"
  if { [info exists arg(n)] } {
    puts $conf "NUM_IONS $arg(n)"
  }
  if { [info exists arg(r)] } {
    puts $conf "R_ION_SOLUTE $arg(r)"
  }
  if { [info exists arg(i)] } {
    puts $conf "R_ION_ION $arg(i)"
  }
  if { [info exists arg(g)] } {
    puts $conf "GRID_STEP $arg(g)"
  }
  if { [info exists arg(b)] } {
    puts $conf "BORDER_WIDTH $arg(b)"
  }
  if { [info exists arg(t)] } {
    puts $conf "ION_NAME $arg(t)"
  }
  if { [info exists arg(q)] } {
    puts $conf "ION_CHARGE $arg(q)"
  }
  if { [info exists arg(p)] } {
    puts $conf "MAX_PROC $arg(p)"
  }
  close $conf

  # write a file with the charges in the beta field
  charge2beta -psf $psffile -pdb $pdbfile -o $pdbcharges

}


proc ::setupionize::charge2beta_usage { } {
  puts "Usage: charge2beta -psf <psf file> -pdb <pdb file> ?-o <output file> (default: charge2beta.pdb)?"
  return
}

proc ::setupionize::charge2beta { args } {

  set nargs [llength $args]
  if { $nargs == 0 || $nargs % 2 } {
    charge2beta_usage
    error ""
  }

  foreach {name val} $args {
    switch -- $name {
      -psf { set arg(psf) $val }
      -pdb { set arg(pdb) $val }
      -o { set arg(o) $val }
      default { error "unknown argument: $name $val" }
    }
  }

  if { ![info exists arg(psf)] } {
    error "Missing psf file."
  } else {
    set psffile $arg(psf)
  }

  if { ![info exists arg(pdb)] } {
    error "Missing pdb file."
  } else {
    set pdbfile $arg(pdb)
  }

  if { ![info exists arg(o)] } {
    set outfile "charge2beta.pdb"
  } else {
    set outfile $arg(o)
  }

  set molid [mol new $psffile type psf]
  mol addfile $pdbfile type pdb molid $molid

  set sel [atomselect $molid all]
  set charge [$sel get charge]
  $sel set beta $charge

  $sel writepdb $outfile
  $sel delete
  mol delete $molid

  return

}

