
/*****************************************************************************/
/*                                                                           */
/* (C) Copyright 2001-2003 Justin Gullingsrud and the University of Illinois.*/
/*                                                                           */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* 
 * Plugin header files; get plugin source from www.ks.uiuc.edu/Research/vmd"
 */
#include "libmolfile_plugin.h"
#include "molfile_plugin.h"

#define CATDCD_MAJOR_VERSION 3
#define CATDCD_MINOR_VERSION 0

/*
 * Read indices specifying which atoms to keep.  Return number of indices,
 * or -1 on error.
 */

static int dcdindices(const char *fname, int **indices) {
  FILE *fd;
  int *ind;
  int num, max;

  max = 10;
  ind = (int *)malloc(max*sizeof(int));
  num = 0;
  fd = fopen(fname, "r");
  if (fd == NULL) {
    fprintf(stderr, "Error opening index file %s\n", fname);
    return -1;
  }
  while (fscanf(fd, "%d", ind+num) == 1) {
    num++;
    if (num == max) {
      max *= 2;
      ind = (int *)realloc(ind, max*sizeof(int));
    }
  }
  fclose(fd);
  *indices = ind;
  return num;
}

static void usage(const char *s) {
  printf("       %s -o outputfile [-i indexfile] [-first firstframe]\n"
         "          [-last lastframe] [-stride stride]\n"
         "          inputfile1 inputfile2 ...\n",s);
  printf("       %s -num inputfile1 inputfile2 ...\n", s);
  exit(1);
}

typedef struct {
  int first;
  int last;
  int stride;
  const char *outfile;
  int num_ind;
  int *ind;
} catdcd_opt_t;

/*
 * Parse args, putting results in opt.  Return the new argc,  
 * or -1 on error.
 */
static int parse_args(int argc, char *argv[], catdcd_opt_t *opt) {
  if (argc < 2) usage(argv[0]);
  argc--;
  argv++;
  while (argc > 1) {
    if (!strcmp(argv[0], "-i")) {
      printf("Reading indices from file '%s'\n", argv[1]);
      opt->num_ind = dcdindices(argv[1], &opt->ind);
      if (opt->num_ind < 0) {
        fprintf(stderr, "Error reading index file.\n");
        return -1;
      }
      if (opt->num_ind < 1) {
        fprintf(stderr, "Error: no indices found in file.\n");
        return -1;
      }
      argc -= 2;
      argv += 2;
      continue;
    } else if (!strcmp(argv[0], "-first")) {
      opt->first = atoi(argv[1]);
      argc -= 2;
      argv += 2;
      continue;
    } else if (!strcmp(argv[0], "-last")) {
      opt->last = atoi(argv[1]);
      argc -= 2;
      argv += 2;
      continue;
    } else if (!strcmp(argv[0], "-stride")) {
      opt->stride = atoi(argv[1]);
      argc -= 2;
      argv += 2;
      continue;
    } else if (!strcmp(argv[0], "-o")) {
      opt->outfile = argv[1];
      argc -= 2;
      argv += 2;
      continue;
    } else if (!strcmp(argv[0], "-num")) {
      /* Silently accept for backward compatibility */
      argc -= 1;
      argv += 1;
      continue;
    }
    /* No flag, so assume everything else is an input file */
    break;
  }
  return argc;
}

static molfile_plugin_t *api;
static int register_cb(void *v, vmdplugin_t *p) {
  api = (molfile_plugin_t *)p;
  return 0;
}

int main(int argc, char *argv[]) {

  catdcd_opt_t opts;
  void *h_in, *h_out;
  int inatoms, outatoms;
  molfile_timestep_t ts_in, ts_out;
  int ifile, rc, frames_read, frames_written;

  printf("CatDCD %d.%d\n", CATDCD_MAJOR_VERSION, CATDCD_MINOR_VERSION);
  opts.first = opts.stride = 1;
  opts.last = -1;
  opts.outfile = NULL;
  opts.num_ind = 0;
  opts.ind = NULL;

  molfile_dcdplugin_init();
  molfile_dcdplugin_register(NULL, register_cb);

  rc = parse_args(argc, argv, &opts);
  if (rc < 0) return 1;
  argv += argc-rc;
  argc = rc;

  if (opts.last != -1 && opts.last < opts.first) {
    fprintf(stderr, "Error: last must be greater than or equal to first.\n");
    return 1;
  }

  /*
   * Peek at the header of the first input file to get the number of atoms.
   * All input files must have this number of atoms. 
   */
  h_in = api->open_file_read(argv[0], "dcd", &inatoms);
  if (!h_in) {
    fprintf(stderr, "Error: could not open file '%s' for reading.\n",
      argv[0]);
    return 1;
  }
  api->close_file_read(h_in);

  /*
   * Open the output file for writing, if there is one
   */
  if (opts.outfile) {
    outatoms = opts.num_ind ? opts.num_ind : inatoms;
    printf("Opening file '%s' for writing.\n", opts.outfile);
    h_out = api->open_file_write(opts.outfile, "dcd", outatoms);
    if (!h_out) {
      fprintf(stderr, "Error: Unable to open output file '%s' for writing.\n",
          opts.outfile);
      return 1;
    }
    ts_in.coords = (float *)malloc(3*inatoms * sizeof(float));
    if (opts.num_ind) {
      ts_out.coords = (float *)malloc(3*outatoms * sizeof(float));
    }
  } else {
    h_out = NULL;
  }


  frames_read = frames_written = 0;
  for (ifile = 0; ifile < argc; ifile++) {
    int tmpatoms;
    int frames_in_file = 0;
    int written_from_file = 0;
    h_in = api->open_file_read(argv[ifile], "dcd", &tmpatoms);
    if (!h_in) {
      fprintf(stderr, "Error: could not open file '%s' for reading.\n",
          argv[ifile]);
      return 1;
    }
    if (tmpatoms != inatoms) {
      fprintf(stderr, 
          "Error: dcd file %s contains wrong number of atoms (%d)\n", 
          argv[ifile], tmpatoms);
      return 1;
    }
    printf("Opened file '%s' for reading.\n", argv[ifile]);
    while (opts.last == -1 || frames_read < opts.last) {
      if (!opts.outfile || frames_read + 1 < opts.first 
          || ((frames_read + 1 - opts.first) % opts.stride)) {
        /* Skip this frame */
        rc = api->read_next_timestep(h_in, inatoms, NULL);
        if (rc == -1) break;
        if (rc < 0) {
          fprintf(stderr, "Error reading input file '%s' (error code %d)\n", 
              argv[ifile], rc);
          return 1;
        }
        frames_read++;
        frames_in_file++;
        continue;
      }
      rc = api->read_next_timestep(h_in, inatoms, &ts_in);
      if (rc == -1) break;
      if (rc < 0) {
        fprintf(stderr, "Error reading input file '%s' (error code %d)\n", 
            argv[ifile], rc);
        return 1;
      }
      frames_read++;
      frames_in_file++;
      if (opts.num_ind) {
        int j;
        for (j=0; j<opts.num_ind; j++) {
          ts_out.coords[3*j] = ts_in.coords[3*opts.ind[j]];
          ts_out.coords[3*j+1] = ts_in.coords[3*opts.ind[j]+1];
          ts_out.coords[3*j+2] = ts_in.coords[3*opts.ind[j]+2];
        }
        ts_out.A = ts_in.A;
        ts_out.B = ts_in.B;
        ts_out.C = ts_in.C;
        ts_out.alpha = ts_in.alpha;
        ts_out.beta = ts_in.beta;
        ts_out.gamma = ts_in.gamma;
        rc = api->write_timestep(h_out, &ts_out);
      } else {
        rc = api->write_timestep(h_out, &ts_in);
      }
      if (rc) {
        fprintf(stderr, "Error writing coordinates frame.\n");
        return 1;
      }
      frames_written++;
      written_from_file++;
    }
    api->close_file_read(h_in);
    printf("Read %d frames from file %s", frames_in_file, argv[ifile]);
    if (opts.outfile) {
      printf(", wrote %d.\n", written_from_file);
    } else {
      printf(".\n");
    }
  }
  printf("Total frames: %d\n", frames_read);
  if (opts.outfile) {
    api->close_file_write(h_out);
    free(ts_in.coords);
    printf("Frames written: %d\n", frames_written);
  }
  if (opts.outfile && opts.num_ind) 
    free(ts_out.coords);
  if (opts.num_ind)
    free(opts.ind);

  printf("CatDCD exited normally.\n");
  return 0;
}

