/* ------------------------------------------- */
/*
 *   Author: James Phillips
 *           Department of Physics
 *           Marquette University
 *           Milwaukee, WI 53233
 *
 *   Purpose: Explorer module which modifies the
 *            coordinates of lattices and pyramids
 *            to produces standard useful flat map
 *            projections and the globe.
 *
 *   Written for the David A. Yuen research group
 *   at the Minnesota Supercomputer Institute and
 *   Department of Geology and Geophysics, 
 *   University of Minnesota.  Summer, 1992.
 *
 */
/* ------------------------------------------- */

#include <cx/DataTypes.h>
#include <cx/DataAccess.h>
#include <stdio.h>
#include <math.h>
#define cot(x) (cos(x)/sin(x))
#define PI 3.1415926535

#define CARTESIAN         0
#define MERCATOR          1
#define LAMBERT_CYL       2
#define GNOMONIC          3
#define STEREO            4
#define ORTHO             5
#define POSTEL            6
#define LAMBERT_AZI       7
#define LAMBERT_CONICAL   8
#define ALBERS            9
#define CASSINI_SOLDNER   10
#define BONNE             11
#define WERNER            12
#define SANSON_FLAMSTEED  13
#define GLOBE             14

void Map_z(float *mapCoord, long nCoord,
           float mapz);

void Cartesian_Unif(float *mapCoord, float globeRadius,
           float paramOne, float paramTwo);

void Cartesian_Perim(float *mapCoord, long xCoord,
           long yCoord, long zCoord, float globeRadius,
           float paramOne, float paramTwo);

void Cartesian_Curvi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Globe(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Mercator_Perim(float *mapCoord, long xCoord,
           long yCoord, long zCoord, float globeRadius,
           float paramOne, float paramTwo);

void Lambert_Curvi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Lambert_Perim(float *mapCoord, long xCoord,
           long yCoord, long zCoord, float globeRadius,
           float paramOne, float paramTwo);

void Mercator_Curvi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Gnomonic(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Stereo(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Ortho(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Postel(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Lambert_Azi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Sanson_Flamsteed(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Cassini_Soldner(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Werner(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo);

void Lambert_Conical(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo,
           float paramThree, float paramFour);

void Albers(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo,
           float paramThree, float paramFour);

void Bonne(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo,
           float paramThree, float paramFour);

int Unif_To_Perimeter(cxCoord **coordIn);

int Perimeter_To_Curvilinear(cxCoord **coordIn);

#ifdef __cplusplus
	extern "C" {
#endif

  void 	projector  (
	 long   	  projType, 
	 double 	  globeRadius, 
	 double 	  mapz, 
	 double 	  paramOne, 
	 double 	  paramTwo, 
	 double 	  paramThree, 
	 double 	  paramFour, 
	 cxPyramid    *	  mapIn, 
	 cxPyramid    *	* mapOut, 
	 cxPyramid    *	  pyr1In, 
	 cxPyramid    *	* pyr1Out, 
	 cxPyramid    *	  pyr2In, 
	 cxPyramid    *	* pyr2Out, 
	 cxPyramid    *	  pyr3In, 
	 cxPyramid    *	* pyr3Out, 
	 cxLattice    *	  lat1In, 
	 cxLattice    *	* lat1Out,
	 cxLattice    *	  lat2In, 
	 cxLattice    *	* lat2Out )
{

   cxPyramid *mapCopy;
   cxLattice *mapLat;
   float *mapCoord;
   long nCoord;
   cxLattice *dataLat;
   cxLattice *tmpLat;
   cxData *tmpData;
   cxConnection *tmpCon;
   cxCoord *tmpCoord;
   long m,n,p,ndims;
   long *dims;
   int i,j,k;
   cxPrimType da;
   cxCoordType db;

   cxLattice *dataCopy;
   cxCoord *coordIn;
   cxCoordType cTypeIn,cTypeOut;
   float *dataCoord;

   cxPyramid *inPyrs[4];
   cxPyramid *outPyrs[4];
   cxLattice *inLats[2];
   cxLattice *outLats[2];
   cxLattice *copyLats[6];
   cxLattice *baseLat;
   int pchanged;

   *mapOut = NULL;
   *pyr1Out = NULL;
   *pyr2Out = NULL;
   *pyr3Out = NULL;
   *lat1Out = NULL;
   *lat2Out = NULL;

   pchanged = (
      cxInputDataChanged(cxInputPortOpen("Projection")) ||
      cxInputDataChanged(cxInputPortOpen("Globe Radius")) ||
      cxInputDataChanged(cxInputPortOpen("Base Long.")) ||
      cxInputDataChanged(cxInputPortOpen("Base Lat.")) ||
      cxInputDataChanged(cxInputPortOpen("1st. St. Par.")) ||
      cxInputDataChanged(cxInputPortOpen("2nd. St. Par."))
   );

   if ( cxInputDataChanged(cxInputPortOpen("Map")) || pchanged ||
      cxInputDataChanged(cxInputPortOpen("Map z")) )
      inPyrs[0] = mapIn; else inPyrs[0] = NULL;
   if ( cxInputDataChanged(cxInputPortOpen("Pyramid 1")) || pchanged )
      inPyrs[1] = pyr1In; else inPyrs[1] = NULL;
   if ( cxInputDataChanged(cxInputPortOpen("Pyramid 2")) || pchanged )
      inPyrs[2] = pyr2In; else inPyrs[2] = NULL;
   if ( cxInputDataChanged(cxInputPortOpen("Pyramid 3")) || pchanged )
      inPyrs[3] = pyr3In; else inPyrs[3] = NULL;
   if ( cxInputDataChanged(cxInputPortOpen("Lattice 1")) || pchanged )
      inLats[0] = lat1In; else inLats[0] = NULL;
   if ( cxInputDataChanged(cxInputPortOpen("Lattice 2")) || pchanged )
      inLats[1] = lat2In; else inLats[1] = NULL;

   /* Copy pyramids. */

   for(i=0;i<4;i++) { if(inPyrs[i]) {
      cxPyrGet(inPyrs[i],&baseLat,&m);
      outPyrs[i] = cxPyrNew(m);
      if(cxDataAllocErrorGet()) {
         for(j=0;j<i;j++) cxDataRefDec(outPyrs[j]);
         return;
      }
      for(n=1;n<=m;n++) {
         cxPyrLayerGet(inPyrs[i],n,&tmpCon,&tmpLat);
         cxPyrLayerSet(outPyrs[i],n,tmpCon,tmpLat);
      }
      tmpLat = baseLat;
      baseLat = cxLatDup(tmpLat,0,1);
      if(cxDataAllocErrorGet()) {
         for(j=0;j<=i;j++) cxDataRefDec(outPyrs[j]);
         return;
      } 
      cxLatPtrGet(tmpLat,&tmpData,NULL,NULL,NULL);
      cxLatPtrSet(baseLat,tmpData,NULL,NULL,NULL);
      cxPyrSet(outPyrs[i],baseLat);
      copyLats[i] = baseLat;
   } else copyLats[i] = NULL, outPyrs[i] = NULL; }

   /* Copy lattices. */

   for(i=0;i<2;i++) { if(inLats[i]) {
      outLats[i] = cxLatDup(inLats[i],0,1);
      if(cxDataAllocErrorGet()) {
         for(j=0;j<4;j++) cxDataRefDec(outPyrs[j]);
         for(j=0;j<i;j++) cxDataRefDec(outLats[j]);
         return;
      } 
      cxLatPtrGet(inLats[i],&tmpData,NULL,NULL,NULL);
      cxLatPtrSet(outLats[i],tmpData,NULL,NULL,NULL);
      copyLats[i+4] = outLats[i];
   } else copyLats[i+4] = NULL, outLats[i] = NULL; }

   /* Transform the coordinates. */

   for(i=0;i<6;i++) if(copyLats[i]) {

      cxLatPtrGet(copyLats[i],NULL,NULL,&coordIn,&dataCoord);
      cxLatDescGet(copyLats[i],&ndims,&dims,
                   NULL,NULL,NULL,
                   NULL,&m,&cTypeIn);

      if(m!=3) { 
         cxModAlert("3 coordinates required.");
         for(j=0;j<4;j++) cxDataRefDec(outPyrs[j]);
         for(j=0;j<2;j++) cxDataRefDec(outLats[j]);
         return;
      }

      /* Change coordinate type if necessary. */

         if ( cTypeIn == cx_coord_uniform && projType != CARTESIAN ) {
            if ( Unif_To_Perimeter(&coordIn) ) {
               for(j=0;j<4;j++) cxDataRefDec(outPyrs[j]);
               for(j=0;j<2;j++) cxDataRefDec(outLats[j]);
               return;
            }
            cxLatPtrSet(copyLats[i],NULL,NULL,coordIn,NULL);
            cTypeIn = cx_coord_perimeter;
         }
         if ( cTypeIn == cx_coord_perimeter &&
              projType != CARTESIAN &&
              projType != LAMBERT_CYL && projType != MERCATOR ) {
            if ( Perimeter_To_Curvilinear(&coordIn) ) {
               for(j=0;j<4;j++) cxDataRefDec(outPyrs[j]);
               for(j=0;j<2;j++) cxDataRefDec(outLats[j]);
               return;
            }
            cxLatPtrSet(copyLats[i],NULL,NULL,coordIn,NULL);
            cTypeIn = cx_coord_curvilinear;
         }

      cxLatPtrGet(copyLats[i],NULL,NULL,&coordIn,&dataCoord);

      nCoord = 1;
      for(j=0;j<ndims;j++) nCoord = nCoord * dims[j];

      if (!i) Map_z(dataCoord,nCoord,mapz);

      /* Modify coordinates. */

      switch( (int)projType ) {

         case CARTESIAN:
            switch(cTypeIn) {
               case cx_coord_uniform:
                  Cartesian_Unif(dataCoord,
                     (float)globeRadius,(float)paramOne,(float)paramTwo);
               break;
               case cx_coord_perimeter:
                  Cartesian_Perim(dataCoord,dims[0],dims[1],dims[2],
                     (float)globeRadius,(float)paramOne,(float)paramTwo);
               break;
               case cx_coord_curvilinear:
                  Cartesian_Curvi(dataCoord,nCoord,
                     (float)globeRadius,(float)paramOne,(float)paramTwo);
               break;
            }
            break;

         case GLOBE:
            Globe(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;

         case MERCATOR:
            switch(cTypeIn) {
               case cx_coord_perimeter:
                  Mercator_Perim(dataCoord,dims[0],dims[1],dims[2],
                     (float)globeRadius,(float)paramOne,(float)paramTwo);
               break;
               case cx_coord_curvilinear:
                  Mercator_Curvi(dataCoord,nCoord,
                     (float)globeRadius,(float)paramOne,(float)paramTwo);
               break;
            }
            break;

         case LAMBERT_CYL:
            switch(cTypeIn) {
               case cx_coord_perimeter:
                  Lambert_Perim(dataCoord,dims[0],dims[1],dims[2],
                     (float)globeRadius,(float)paramOne,(float)paramTwo);
               break;
               case cx_coord_curvilinear:
                  Lambert_Curvi(dataCoord,nCoord,
                     (float)globeRadius,(float)paramOne,(float)paramTwo);
               break;
            }
            break;

         case GNOMONIC:
            Gnomonic(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case STEREO:
            Stereo(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case ORTHO:
            Ortho(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case POSTEL:
            Postel(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case LAMBERT_AZI:
            Lambert_Azi(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case SANSON_FLAMSTEED:
            Sanson_Flamsteed(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case CASSINI_SOLDNER:
            Cassini_Soldner(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case WERNER:
            Werner(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo);
            break;
         case LAMBERT_CONICAL:
            Lambert_Conical(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo,
                  (float)paramThree,(float)paramFour);
            break;
         case ALBERS:
            Albers(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo,
                  (float)paramThree,(float)paramFour);
            break;
         case BONNE:
            Bonne(dataCoord,nCoord,(float)globeRadius,
                  (float)paramOne,(float)paramTwo,
                  (float)paramThree,(float)paramFour);
            break;

      }
   
   }

   *mapOut = outPyrs[0];
   *pyr1Out = outPyrs[1];
   *pyr2Out = outPyrs[2];
   *pyr3Out = outPyrs[3];
   *lat1Out = outLats[0];
   *lat2Out = outLats[1];

   return;

}

#ifdef __cplusplus
}
#endif
/* ------------------------------------------- */

void Map_z(float *mapCoord, long nCoord,
           float mapz)
{

   long i;
   float *coord;

   coord = mapCoord;

   for(i=0;i<nCoord;i++) {
      coord[2] += mapz;
      coord += 3;
   }

}

void Globe(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;

   coord = mapCoord;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*3.1415926535/180;
      y = coord[1]*3.1415926535/180;
      z = coord[2];
      coord[0] = ( globeRadius + z ) * cos(y) * cos(x);
      coord[1] = ( globeRadius + z ) * cos(y) * sin(x);
      coord[2] = ( globeRadius + z ) * sin(y);
      coord += 3;
   }

}

void Cartesian_Unif(float *mapCoord, float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float xmin,xmax,ymin,ymax;

   coord = mapCoord;

   xmin = (coord[0]-paramOne)*3.1415926535/180;
   xmax = (coord[1]-paramOne)*3.1415926535/180;
   ymin = (coord[2]-paramTwo)*3.1415926535/180;
   ymax = (coord[3]-paramTwo)*3.1415926535/180;
   coord[0] = globeRadius * xmin;
   coord[1] = globeRadius * xmax;
   coord[2] = globeRadius * ymin;
   coord[3] = globeRadius * ymax;

}

void Cartesian_Perim(float *mapCoord, long xCoord,
           long yCoord, long zCoord, float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;

   coord = mapCoord;

   for(i=0;i<xCoord;i++) {
      x = (coord[0]-paramOne)*3.1415926535/180;
      coord[0] = globeRadius * x;
      coord ++;
   }
   for(i=0;i<yCoord;i++) {
      y = (coord[0]-paramTwo)*3.1415926535/180;
      coord[0] = globeRadius * y;
      coord ++;
   }

}

void Cartesian_Curvi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z,xo;

   coord = mapCoord;

   for(i=0;i<nCoord;i++) {
      x = (coord[0]-paramOne)*3.1415926535/180;
      y = (coord[1]-paramTwo)*3.1415926535/180;
      z = coord[2];
      coord[0] = globeRadius * x;
      coord[1] = globeRadius * y;
      coord[2] = z;
      coord += 3;
   }

}

void Mercator_Perim(float *mapCoord, long xCoord,
           long yCoord, long zCoord, float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z,xo,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;
   yo = globeRadius * log((1+sin(yo))/cos(yo));

   for(i=0;i<xCoord;i++) {
      x = coord[0]*3.1415926535/180;
      coord[0] = globeRadius * (x-xo) ;
      coord ++;
   }
   for(i=0;i<yCoord;i++) {
      y = coord[0]*3.1415926535/180;
      coord[0] = globeRadius * log((1+sin(y))/cos(y)) - yo;
      coord ++;
   }

}

void Mercator_Curvi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z,xo,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;
   yo = globeRadius * log((1+sin(yo))/cos(yo));

   for(i=0;i<nCoord;i++) {
      x = coord[0]*3.1415926535/180;
      y = coord[1]*3.1415926535/180;
      z = coord[2];
      coord[0] = globeRadius * (x-xo);
      coord[1] = globeRadius * log((1+sin(y))/cos(y)) - yo;
      coord[2] = z;
      coord += 3;
   }

}

void Lambert_Perim(float *mapCoord, long xCoord,
           long yCoord, long zCoord, float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z,xo,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;
   yo = globeRadius * sin(yo);

   for(i=0;i<xCoord;i++) {
      x = coord[0]*3.1415926535/180;
      coord[0] = globeRadius * (x-xo) ;
      coord ++;
   }
   for(i=0;i<yCoord;i++) {
      y = coord[0]*3.1415926535/180;
      coord[0] = globeRadius * sin(y) - yo;
      coord ++;
   }

}

void Lambert_Curvi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z,xo,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;
   yo = globeRadius * sin(yo);

   for(i=0;i<nCoord;i++) {
      x = coord[0]*3.1415926535/180;
      y = coord[1]*3.1415926535/180;
      z = coord[2];
      coord[0] = globeRadius * (x-xo);
      coord[1] = globeRadius * sin(y) - yo;
      coord[2] = z;
      coord += 3;
   }

}

void Gnomonic(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;
   float xo,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*PI/180;
      y = coord[1]*PI/180;
      z = coord[2];
      coord[0] = globeRadius * cos(y) * sin(x-xo) / 
                 ( sin(yo)*sin(y) + cos(yo)*cos(y)*cos(x-xo) );
      coord[1] = globeRadius * ( cos(yo)*sin(y) - sin(yo)*cos(y)*cos(x-xo) ) /
                 ( sin(yo)*sin(y) + cos(yo)*cos(y)*cos(x-xo) );
      coord[2] = z;
      coord += 3;
   }

}

void Stereo(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;
   float xo,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*PI/180;
      y = coord[1]*PI/180;
      z = coord[2];
      coord[0] = globeRadius * 2.0 * cos(y) * sin(x-xo) /
                 ( 1 + sin(yo)*sin(y) + cos(yo)*cos(y)*cos(x-xo) );
      coord[1] = globeRadius * 2.0 * ( cos(yo)*sin(y) - sin(yo)*cos(y)*cos(x-xo) ) /
                 ( 1 + sin(yo)*sin(y) + cos(yo)*cos(y)*cos(x-xo) );
      coord[2] = z;
      coord += 3;
   }

}

void Ortho(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;
   float xo,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*PI/180;
      y = coord[1]*PI/180;
      z = coord[2];
      coord[0] = globeRadius * cos(y) * sin(x-xo);
      coord[1] = globeRadius * ( cos(yo)*sin(y) - sin(yo)*cos(y)*cos(x-xo) );
      coord[2] = z;
      coord += 3;
   }

}

void Postel(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;
   float xo,yo;
   float ic,jc,m,d;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*PI/180;
      y = coord[1]*PI/180;
      d = acos(cos(x-xo)*cos(y)*cos(yo) + sin(y)*sin(yo));
      ic = cos(y)*sin(x-xo);
      jc = cos(yo)*sin(y) - cos(x-xo)*cos(y)*sin(yo);
      m = sqrt(pow(ic,2)+pow(jc,2));
      if (m==0.0) m = 1.0;
      ic = ic/m;
      jc = jc/m;
      coord[0] = globeRadius * d * ic;
      coord[1] = globeRadius * d * jc;
      coord += 3;
   }

}

void Lambert_Azi(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;
   float xo,yo;
   float ic,jc,m,d;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*PI/180;
      y = coord[1]*PI/180;
      d = acos(cos(x-xo)*cos(y)*cos(yo) + sin(y)*sin(yo));
      ic = cos(y)*sin(x-xo);
      jc = cos(yo)*sin(y) - cos(x-xo)*cos(y)*sin(yo);
      m = sqrt(pow(ic,2)+pow(jc,2));
      if (m==0.0) m = 1.0;
      ic = ic/m;
      jc = jc/m;
      coord[0] = globeRadius * 2 * sin(d/2) * ic;
      coord[1] = globeRadius * 2 * sin(d/2) * jc;
      coord += 3;
   }

}

void Sanson_Flamsteed(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z,yo;

   coord = mapCoord;
   yo = paramTwo*PI/180;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*3.1415926535/180;
      y = coord[1]*3.1415926535/180;
      z = coord[2];
      coord[0] = globeRadius * (x-paramOne*PI/180) * cos(y);
      coord[1] = globeRadius * (y-yo);
      coord[2] = z;
      coord += 3;
   }

}

void Cassini_Soldner(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;
   float xo,dx,a,t,yo;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;
   yo = globeRadius * yo;

   for(i=0;i<nCoord;i++) {
      dx = coord[0]*PI/180 - xo;
      y = coord[1]*PI/180;
      a = dx * cos(y);
      t = tan(y);
      coord[0] = globeRadius * ( a - pow(a,3)*pow(t,2)/6 -
                   pow(a,5)*pow(t,2)*(8-pow(t,2))/120 );
      coord[1] = globeRadius * ( y + pow(a,2)*t/2 +
                   pow(a,4)*t*(5-pow(t,2))/24 ) - yo;
      coord += 3;
   }

}

void Werner(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo)
{

   long i;
   float *coord;
   float x,y,z;
   float f,yo;

   coord = mapCoord;
   yo = paramTwo*PI/180;

   if(paramTwo>=0.0) f = 1; else f = -1;
   yo = globeRadius * -1 * (PI/2*f-yo);

   for(i=0;i<nCoord;i++) {
      x = coord[0]*3.1415926535/180;
      y = coord[1]*3.1415926535/180;
      coord[0] = globeRadius * (PI/2-y*f) * sin((x-paramOne*PI/180)*cos(y)/(PI/2-y*f));
      coord[1] = globeRadius * -1 * (PI/2*f-y) * cos((x-paramOne*PI/180)*cos(y)/(PI/2-y*f)) - yo;
      coord += 3;
   }

}

void Lambert_Conical(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo,
           float paramThree, float paramFour)
{

   long i;
   float *coord;
   float x,xo,y,yo,z,s;
   float p,po,pa,pb;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;
   pa = paramThree*PI/180;
   pb = paramFour*PI/180;

   if ( paramThree == paramFour ) {
      s = sin(pa);
      po = globeRadius * cot(yo);
      p = po * powf(tan(PI/4-y/2)/tan(PI/4-pa/2),s);
      if(tan(PI/4-y/2)/tan(PI/4-pa/2)<0.0)p=0.0;
      yo = -1 * p;

      for(i=0;i<nCoord;i++) {
         x = coord[0]*PI/180;
         y = coord[1]*PI/180;
         p = po * powf(tan(PI/4-y/2)/tan(PI/4-pa/2),s);
         if(tan(PI/4-y/2)/tan(PI/4-pa/2)<0.0)p=0.0;
         coord[0] = p * sin((x-xo)*s);
         coord[1] = -1 * p * cos((x-xo)*s) - yo;
         coord += 3;
      }
   }
   else {
      s = (log(cos(pa))-log(cos(pb)))/
        (log(tan(PI/4-pa/2))-log(tan(PI/4-pb/2)));
      po = globeRadius * cot(yo);
      p = globeRadius * ( cos(pa) / s ) *
          powf(tan(PI/4-yo/2)/tan(PI/4-pa/2),s);
      if(tan(PI/4-y/2)/tan(PI/4-pa/2)<0.0)p=0.0;
      yo = -1 * p;

      for(i=0;i<nCoord;i++) {
         x = coord[0]*PI/180;
         y = coord[1]*PI/180;
         p = globeRadius * ( cos(pa) / s ) *
             powf(tan(PI/4-y/2)/tan(PI/4-pa/2),s);
         if(tan(PI/4-y/2)/tan(PI/4-pa/2)<0.0)p=0.0;
         coord[0] = p * sin((x-xo)*s);
         coord[1] = -1 * p * cos((x-xo)*s) - yo;
         coord += 3;
      }
   }

}

void Albers(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo,
           float paramThree, float paramFour)
{

   long i;
   float *coord;
   float x,xo,y,yo,z;
   float p,po,pa,pb;

   coord = mapCoord;
   xo = paramOne*PI/180;
   yo = paramTwo*PI/180;
   pa = paramThree*PI/180;
   pb = paramFour*PI/180;

   if ( paramThree == paramFour ) {
      po = globeRadius * cot(yo);
      p = sqrt(1+pow(sin(pa),2)-2*sin(yo)*sin(pa));
      yo = globeRadius * ( cos(pa) - p ) * cos((x-xo)*sin(pa)) / sin(pa);

      for(i=0;i<nCoord;i++) {
         x = coord[0]*PI/180;
         y = coord[1]*PI/180;
         p = sqrt(1+pow(sin(pa),2)-2*sin(y)*sin(pa));
         coord[0] = globeRadius * p * sin((x-xo)*sin(pa)) / sin(pa);
         coord[1] = globeRadius * ( cos(pa) - p ) * cos((x-xo)*sin(pa)) / sin(pa);
         coord += 3;
      }
   }
   else {
      po = globeRadius * cot(yo);
      p = 2 * globeRadius * sqrt(pow(cos(pa),2)+
          (sin(pa)-sin(yo))*(sin(pa)+sin(pb))) / (sin(pa)+sin(pb));
      yo = po - p;

      for(i=0;i<nCoord;i++) {
         x = coord[0]*PI/180;
         y = coord[1]*PI/180;
         p = 2 * globeRadius * sqrt(pow(cos(pa),2)+
          (sin(pa)-sin(y))*(sin(pa)+sin(pb))) / (sin(pa)+sin(pb));
         coord[0] = p * sin((x-xo)*(sin(pa)+sin(pb))/2);
         coord[1] = po - p * cos((x-xo)*(sin(pa)+sin(pb))/2) - yo;
         coord += 3;
      }
   }

}

void Bonne(float *mapCoord, long nCoord,
           float globeRadius,
           float paramOne, float paramTwo,
           float paramThree, float paramFour)
{

   long i;
   float *coord;
   float x,xo,y,z;
   float p,po,pa;

   coord = mapCoord;
   xo = paramOne*PI/180;
   pa = paramTwo*PI/180;

   for(i=0;i<nCoord;i++) {
      x = coord[0]*PI/180;
      y = coord[1]*PI/180;
      p = cot(pa) - (y-pa);
      coord[0] = globeRadius * p * sin( (x-xo)*cos(y)/p );
      coord[1] = globeRadius * ( cot(pa) - p * cos( (x-xo)*cos(y)/p ) );
      coord += 3;
   }

}

int Unif_To_Perimeter(cxCoord **coordParam)
{

   cxCoord *coordIn;
   cxCoord *coordOut;
   float *cIn;
   float *cOut;
   int i,j;

   coordIn = *coordParam;

   coordOut = cxCoordNew(3,coordIn->dims,3,cx_coord_perimeter);
   if(cxDataAllocErrorGet()) return 1;
   cIn = cxCoordValsGet(coordIn);
   cOut = cxCoordValsGet(coordOut);

   for(i=0;i<3;i++) {
      for(j=0;j<coordIn->dims[i];j++) {
         cOut[0] = (cIn[1]-cIn[0])*((float)j)/
               ((float)(coordIn->dims[i]-1)) + cIn[0];
         cOut++;
      }
      cIn += 2;
   }

   *coordParam = coordOut;
   return 0;

}

int Perimeter_To_Curvilinear(cxCoord **coordParam)
{

   cxCoord *coordIn;
   cxCoord *coordOut;
   float *cIn;
   float *ci;
   float *cj;
   float *ck;
   float *cOut;
   int i,j,k;

   coordIn = *coordParam;

   coordOut = cxCoordNew(3,coordIn->dims,3,cx_coord_curvilinear);
   if(cxDataAllocErrorGet()) return 1;
   cIn = cxCoordValsGet(coordIn);
   ci = cIn;
   cj = cIn + coordIn->dims[0];
   ck = cIn + coordIn->dims[0] + coordIn->dims[1];
   cOut = cxCoordValsGet(coordOut);

   for(k=0;k<coordIn->dims[2];k++)
      for(j=0;j<coordIn->dims[1];j++)
         for(i=0;i<coordIn->dims[0];i++) {
            cOut[0] = ci[i];
            cOut[1] = cj[j];
            cOut[2] = ck[k];
            cOut += 3;
         }

   *coordParam = coordOut;
   return 0;

}





