/* ------------------------------------------- */
/*
 *   Author: James Phillips
 *           Department of Physics
 *           Marquette University
 *           Milwaukee, WI 53233
 *
 *   Purpose: Explorer module which creates a
 *            pyramid structure containing an
 *            enhanced bounding box.
 *
 *   Written for the David A. Yuen research group
 *   at the Minnesota Supercomputer Institute and
 *   Department of Geology and Geophysics, 
 *   University of Minnesota.  Summer, 1992.
 *
 */
/* ------------------------------------------- */
#include <cx/DataTypes.h>
#include <cx/DataAccess.h>
#include <cx/UserFuncs.h>
#include <math.h>

#define NONE 0
#define FULL 1
#define TICKS 2

#define SMX 1.0
#define SMY 1.0
#define SMZ 100.0

   void BrokenLine(float *coords, long *elements, long *connections,
                   long *npoints, long *nsegments,
                   float xa, float xb,
                   float ya, float yb,
                   float za, float zb,
                   float sm );


#ifdef __cplusplus
	extern "C" {
#endif

  void 	bendbox  (
	 long   	  x_grid, 
	 long   	  y_grid, 
	 long   	  z_grid, 
	 float  	  x_spacing, 
	 float  	  y_spacing, 
	 float  	  z_spacing, 
	 float  	  x_min, 
	 float  	  x_max, 
	 float   	  y_min, 
	 float   	  y_max, 
	 float   	  z_min, 
	 float   	  z_max, 
	 cxPyramid    *	* box )
{

   long i,j,k,dims[1],npoints,nsegments,onp,ons;
   long nx,ny,nz,bx,by,bz,tbx,tby,tbz;
   float x,y,z,xt,yt,zt;
   cxPyramid *theBox;
   cxLattice *segments,*points;
   cxConnection *conns;
   long *elements,*connections;
   float x_gmin,y_gmin,z_gmin;
   float *colors,*coords;

   *box = NULL;

   if ( x_min>x_max || y_min>y_max || z_min>z_max ||
        x_spacing <= 0 || y_spacing <= 0 || z_spacing <= 0 )
      return;

   /* Find size of box. */
   xt = (x_max-x_min)/60.0;
   yt = (y_max-y_min)/60.0;
   zt = (z_max-z_min)/60.0;
   nx=0;ny=0;nz=0;
   x_gmin = x_spacing*floor(x_min/x_spacing);
   y_gmin = y_spacing*floor(y_min/y_spacing);
   z_gmin = z_spacing*floor(z_min/z_spacing);
   if(x_grid)for(x=x_gmin+x_spacing;x<x_max;x+=x_spacing,nx++);
   if(y_grid)for(y=y_gmin+y_spacing;y<y_max;y+=y_spacing,ny++);
   if(z_grid)for(z=z_gmin+z_spacing;z<z_max;z+=z_spacing,nz++);
   for(x=x_min+SMX,bx=1;x<x_max;x+=SMX,bx++);
   for(y=y_min+SMY,by=1;y<y_max;y+=SMY,by++);
   for(z=z_min+SMZ,bz=1;z<z_max;z+=SMZ,bz++);
   for(x=SMX,tbx=1;x<xt;x+=SMX,tbx++);
   for(y=SMY,tby=1;y<yt;y+=SMY,tby++);
   for(z=SMZ,tbz=1;z<zt;z+=SMZ,tbz++);

   npoints = 0;
   nsegments = 0;
   npoints+=(bz+1)*4; nsegments+=(bz)*4;
   npoints+=(by+1)*4; nsegments+=(by)*4;
   npoints+=(bx+1)*4; nsegments+=(bx)*4;
   if(x_grid==FULL) {
      npoints+=nx*(2*(by+1)+2*(bz+1));
      nsegments+=nx*(2*by+2*bz);
   }
   else if(x_grid==TICKS) {
      npoints+=nx*2*(2*(tby+1)+2*(tbz+1));
      nsegments+=nx*2*(2*tby+2*tbz);
   }
   if(y_grid==FULL) {
      npoints+=ny*(2*(bx+1)+2*(bz+1));
      nsegments+=ny*(2*bx+2*bz);
   }
   else if(y_grid==TICKS) {
      npoints+=ny*2*(2*(tbx+1)+2*(tbz+1));
      nsegments+=ny*2*(2*tbx+2*tbz);
   }
   if(z_grid==FULL) {
      npoints+=nz*(2*(by+1)+2*(bx+1));
      nsegments+=nz*(2*by+2*bx);
   }
   else if(z_grid==TICKS) {
      npoints+=nz*2*(2*(tby+1)+2*(tbx+1));
      nsegments+=nz*2*(2*tby+2*tbx);
   }
   
   onp = npoints; ons = nsegments;

   /* Allocate box. */

   theBox = cxPyrNew(1);
   dims[0] = npoints;
   points = cxLatNew(1,dims,1,cx_prim_float,3,cx_coord_curvilinear);
   dims[0] = nsegments;
   segments = cxLatRootNew(1,dims);
   conns = cxConnNew(nsegments,2*nsegments);
   if (cxDataAllocErrorGet()) {
      cxModAlert("Out of shared memory!");
      cxDataRefDec(points);
      cxDataRefDec(conns);
      cxDataRefDec(segments);
      cxDataRefDec(theBox);
      return;
   }
   cxLatPtrGet(points,NULL,&colors,NULL,&coords);
   cxConnPtrGet(conns,NULL,NULL,&elements,&connections);
   elements[nsegments] = 2*nsegments;
   cxPyrSet(theBox,points);
   cxPyrLayerSet(theBox,1,conns,segments);

   /* Build box. */

/*
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y,y,z,z,SMX);
*/

   npoints = 0;
   nsegments = 0;

      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y_min,y_min,z_min,z_min,SMX);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y_min,y_min,z_max,z_max,SMX);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y_max,y_max,z_min,z_min,SMX);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y_max,y_max,z_max,z_max,SMX);

      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y_min,y_max,z_min,z_min,SMY);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y_min,y_max,z_max,z_max,SMY);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y_min,y_max,z_min,z_min,SMY);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y_min,y_max,z_max,z_max,SMY);

      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y_min,y_min,z_min,z_max,SMZ);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y_max,y_max,z_min,z_max,SMZ);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y_min,y_min,z_min,z_max,SMZ);
      BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y_max,y_max,z_min,z_max,SMZ);

   if(x_grid==FULL)
      for(x=x_gmin+x_spacing;x<x_max;x+=x_spacing) {
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_min,y_min,z_min,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_max,y_max,z_min,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_min,y_max,z_min,z_min,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_min,y_max,z_max,z_max,SMY);
      }
   else if(x_grid==TICKS)
      for(x=x_gmin+x_spacing;x<x_max;x+=x_spacing) {
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_min,y_min,z_min,z_min+zt,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_max,y_max,z_min,z_min+zt,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_min,y_min+yt,z_min,z_min,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_min,y_min+yt,z_max,z_max,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_min,y_min,z_max-zt,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_max,y_max,z_max-zt,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_max-yt,y_max,z_min,z_min,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x,x,y_max-yt,y_max,z_max,z_max,SMY);
      }
   if(y_grid==FULL)
      for(y=y_gmin+y_spacing;y<y_max;y+=y_spacing) {
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y,y,z_min,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y,y,z_min,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y,y,z_min,z_min,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y,y,z_max,z_max,SMX);
      }
   else if(y_grid==TICKS)
      for(y=y_gmin+y_spacing;y<y_max;y+=y_spacing) {
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y,y,z_min,z_min+zt,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y,y,z_min,z_min+zt,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min+xt,y,y,z_min,z_min,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min+xt,y,y,z_max,z_max,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y,y,z_max-zt,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y,y,z_max-zt,z_max,SMZ);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max-xt,x_max,y,y,z_min,z_min,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max-xt,x_max,y,y,z_max,z_max,SMX);
      }
   if(z_grid==FULL)
      for(z=z_gmin+z_spacing;z<z_max;z+=z_spacing) {
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y_min,y_min,z,z,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_max,y_max,y_max,z,z,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y_min,y_max,z,z,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y_min,y_max,z,z,SMY);
      }
   else if(z_grid==TICKS)
      for(z=z_gmin+z_spacing;z<z_max;z+=z_spacing) {
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min+xt,y_min,y_min,z,z,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min+xt,y_max,y_max,z,z,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y_min,y_min+yt,z,z,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y_min,y_min+yt,z,z,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max-xt,x_max,y_min,y_min,z,z,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max-xt,x_max,y_max,y_max,z,z,SMX);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_min,x_min,y_max-yt,y_max,z,z,SMY);
         BrokenLine(coords,elements,connections,&npoints,&nsegments,
                 x_max,x_max,y_max-yt,y_max,z,z,SMY);
      }

   if (onp!=npoints) {
      cxModAlert("Hmm. Seems to be a bug in here.\nPoint count is off.");
      printf("%ld vs. %ld\n",onp,npoints);
      cxDataRefDec(theBox);
      return;
   }
   if (ons!=nsegments) {
      cxModAlert("Hmm. Seems to be a bug in here.\nSegment count is off.");
      printf("%ld vs. %ld\n",ons,npoints);
      cxDataRefDec(theBox);
      return;
   }

   /* Add color to box. */

   for(i=0;i<npoints;i++) colors[i] = 99.0;
   
   /* Output box and return. */

   *box = theBox;
   return;

}

#ifdef __cplusplus
}
#endif
/* ------------------------------------------- */



   void BrokenLine(float *coords, long *elements, long *connections,
                   long *npoints, long *nsegments,
                   float xa, float xb,
                   float ya, float yb,
                   float za, float zb,
                   float sm )
{

   long np,ns;
   float i,j,k,m,ms;
   float x,y,z;

   np = *npoints;
   ns = *nsegments;

   i = xb - xa; j = yb - ya; k = zb - za;
   ms = i*i+j*j+k*k;
   m = fsqrt(ms);
   if ( m == 0.0 ) m = 1.0;
   i = i/m; j = j/m; k = k/m;

   coords[3*np+0] = xa;
   coords[3*np+1] = ya;
   coords[3*np+2] = za;
   connections[2*ns+0] = np;
   connections[2*ns+1] = np + 1;
   elements[ns] = 2*ns;
   np++;
   ns++;

   for(x=sm*i,y=sm*j,z=sm*k;(x*x+y*y+z*z)<ms;x+=sm*i,y+=sm*j,z+=sm*k) {
      coords[3*np+0] = xa + x;
      coords[3*np+1] = ya + y;
      coords[3*np+2] = za + z;
      connections[2*ns+0] = np;
      connections[2*ns+1] = np + 1;
      elements[ns] = 2*ns;
      np++;
      ns++;
   }

   coords[3*np+0] = xb;
   coords[3*np+1] = yb;
   coords[3*np+2] = zb;
   np++;

   *npoints = np;
   *nsegments = ns;
   return;

}




